<?php
/**
 * Plugin Name:         ActivityPub Auto Mention
 * Description:         Automatically mention selected Fediverse profiles on all posts
 * Plugin URI:          https://plugins.seindal.dk/plugins/activitypub-auto-mention/
 * Update URI:          https://plugins.seindal.dk
 * Author:              René Seindal
 * Author URI:          https://plugins.seindal.dk/
 * Donate link:         https://mypos.com/@historywalks
 * Requires Plugins:    activitypub, rs-base-plugin
 * License:             GPL v2 or later
 * License URI:         https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:         activitypub-auto-mention
 * Domain Path:         /languages
 * Requires PHP:        7.4
 * Requires at least:   5.0
 * Version:             1.1.11
 **/

namespace ReneSeindal;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

add_action( 'plugins_loaded', function() {
    class AP_AutoMentions extends PluginBase {
        protected $plugin_file = __FILE__;

        use PluginBaseSettings;

        /*************************************************************
         *
         * ActivityPub adaptations
         *
         *************************************************************/

        // Convert text field to sanitized array
        function get_mentions( $mentions = NULL ) {
            if ( empty( $mentions ) )
                $mentions = $this->get_option( 'mentions' );

            return array_filter(
                array_map(
                    fn( $line ) => trim( $line ),
                    explode( "\n", $mentions ?: '' )
                ),
                fn ( $line ) => !empty( $line )
            );
        }

        function do_activitypub_extract_mentions_filter( $mentions, $post_content ) {
            if ( class_exists( '\Activitypub\Webfinger' ) ) {
                $auto_mentions = $this->get_mentions();

                foreach ( $auto_mentions as $mention ) {
                    $link = $this->webfinger_resolve( $mention );
                    if ( $link )
                        $mentions[ $mention ] = $link;
                }
            }

            return $mentions;
        }

        function webfinger_resolve( $mention ) {
            if ( ! class_exists( '\Activitypub\Webfinger' ) )
                return false;

            $link = \Activitypub\Webfinger::resolve( $mention );

            if ( is_wp_error( $link ) )
                return false;

            return $link;
        }

        /************************************************************************
         *
         *	Helper shortcodes
         *
         ************************************************************************/

        // [ap_content_short] -- like [ap_content] but truncates the
        // content at the --read more-- separator, if present.

        function do_ap_content_short_shortcode( $atts = [], $content = null, $tag = '' ) {
            // normalize attribute keys, lowercase
            $atts = array_change_key_case( (array)$atts, CASE_LOWER );

            // override default attributes with user attributes
            $atts = shortcode_atts( [
                'apply_filters' => 'yes',
            ] , $atts, $tag );

            $post = $this->shortcode_get_post();
            if ( empty( $post ) )
                return '';

            $content = $post->post_content;

            if ( ! str_contains( $content, '<!--more-->' ) )
                return do_shortcode( '[ap_content]');

            list( $content, $rest ) = explode( '<!--more-->', $content, 2 );

            return $this->cleanup_shortcode_output( $content );
        }


        // [ap_excerpt_long] -- like [ap_excerpt] but if the main
        // content has a --read more-- separator, it uses the first
        // part of the content

        function do_ap_excerpt_long_shortcode( $atts = [], $content = null, $tag = '' ) {
            // normalize attribute keys, lowercase
            $atts = array_change_key_case( (array)$atts, CASE_LOWER );

            // override default attributes with user attributes
            $atts = shortcode_atts( [
                'depth' => 1,
            ] , $atts, $tag );

            $post = $this->shortcode_get_post();
            if ( empty( $post ) )
                return '';

            $content = $post->post_content;

            if ( ! str_contains( $content, '<!--more-->' ) )
                return do_shortcode( '[ap_excerpt]');

            list( $content, $rest ) = explode( '<!--more-->', $content, 2 );
            return $this->cleanup_shortcode_output( $content );
        }

        private function shortcode_get_post() {
            $post = get_post();

            if ( empty( $post ) )
                return NULL;
            if ( ! is_post_publicly_viewable( $post ) )
                return NULL;
            if ( post_password_required( $post ) )
                return NULL;

            return $post;
        }

        private function cleanup_shortcode_output( $content ) {
            $content = \apply_filters( 'the_content', $content );

            // replace script and style elements
            $content = \preg_replace( '@<(script|style)[^>]*?>.*?</\\1>@si', '', $content );
            $content = strip_shortcodes( $content );
            $content = \trim( \preg_replace( '/[\n\r\t]/', '', $content ) );

            return $content;
        }


        /************************************************************
         *
         * Generate a longer excerpt of a post with limited markup.
         *
         * It extracts leading paragraphs from the content, outputting
         * always a set of whole paragraphs, within the character
         * limit of $max_length.
         *
         * Processing stops at any unrecognised tag, such as
         * headers, lists, blockquotes etc., and at the <!--more-->
         * marker to allow manual intervention.
         *
         ************************************************************/

        function generate_excerpt( $post, $max_length = 1000 ) {
            $post = get_post( $post );

            // Expand in-text shortcodes but not blocks/patterns
            $input = do_shortcode( $post->post_content );

            $paragraphs = [];        // found paragraphs
            $buffer = '';       // output buffer

            $done = false;

            $parser = new \WP_HTML_TAG_Processor( $input );
            while ( !$done && $parser->next_token() ) {

                switch ( $parser->get_token_type() ) {
                case '#text':
                    $buffer .= $parser->get_modifiable_text();
                    break;

                case '#comment':
                    if ( 'more' === $parser->get_modifiable_text() )
                        $done = true;
                    break;

                case '#tag':
                    $tag = $parser->get_token_name();

                    switch ( $tag ) {
                    case 'I':   // Copy along
                    case 'EM':
                    case 'B':
                    case 'STRONG':
                        if ( $parser->is_tag_closer() )
                            $buffer .= "</$tag>";
                        else
                            $buffer .= "<$tag>";
                        break;

                    case 'BR':
                        if ( ! $parser->is_tag_closer() )
                            $buffer .= "<$tag/>";
                        break;

                    case 'A':   // Copy with href
                        if ( $parser->is_tag_closer() )
                            $buffer .= "</$tag>";
                        else {
                            $href = $parser->get_attribute( 'href' );
                            $buffer .= sprintf( '<A HREF="%s">', $href ?? '' );
                        }
                        break;

                    case 'SUP':
                    case 'FIGURE':
                        // Footnote and figure markup -- ignore
                        // everything up to the closing tag -- This is
                        // very primitive - sorry!
                        if ( $parser->is_tag_closer() )
                            $buffer = $stashed;
                        else
                            $stashed = $buffer;
                        break;

                    case 'IMG':
                    case 'FIGCAPTION':
                        // Images -- skip the markup
                        break;

                    case 'BDO':
                        // Text in other languages -- skip the markup
                        break;

                    case 'P':
                    case 'BLOCKQUOTE':
                    case 'PRE': // PRE & BLOCKQUOTE become P in the excerpt
                        if ( $parser->is_tag_closer() ) {
                            // Take into account <P> </P> and a newline
                            $max_length -= strlen( $buffer ) + 6 + 2*strlen( $tag );
                            if ( $max_length >= 0 )
                                $paragraphs[] = trim( $buffer );
                            else
                                $done = true;

                            $buffer = '';
                        }
                        break;

                    default:
                        // Unknown markup will stop processing, but
                        // also ditch the partially processed
                        // paragraph. This might not be optimal, but
                        // it resolves the problem of having
                        // surprising breaks in the text.

                        $done = true;
                    }
                }
            }

            if ( empty( $paragraphs ) )
                return get_the_excerpt( $post );

             return '<P>' . join( "</P><P>", $paragraphs ) . '</P>';
        }


        function do_ap_intro_shortcode( $atts = [], $content = null, $tag = '' ) {
            $post = $this->shortcode_get_post();
            if ( empty( $post ) )
                return '';

            return $this->generate_excerpt( $post );
        }



        /************************************************************************
         *
         *	Settings
         *
         ************************************************************************/

        function do_admin_menu_action() {
            $this->settings_add_menu(
                __( 'ActivityPub Auto Mentions', 'activitypub-auto-mention' ),
                __( 'AP Auto Mentions', 'activitypub-auto-mention' )
            );
        }

        // Called automatically by PluginBaseSettings
        function settings_define_sections_and_fields() {
            $section = $this->settings_add_section(
                'section_mentions',
                __( 'Auto mentions', 'activitypub-auto-mention' ),
                __( 'List the Fediverse handles you want to mention always, one on each line.', 'activitypub-auto-mention' )
            );

            $this->settings_add_field(
                'mentions', $section,
                __( 'Mentions', 'activitypub-auto-mention' ),
                'settings_field_textarea_html'
            );
        }

        // Called automatically by PluginBaseSettings
        function settings_sanitize_option( $setting ) {
            $mentions = $this->get_mentions( $setting['mentions'] );

            // Check then all to report errors to the user
            foreach ( $mentions as $mention ) {
                if ( ! $this->webfinger_resolve( $mention ) )
                    add_settings_error(
                        $this->option_name,
                        sanitize_key( $mention ),
                        sprintf( __( 'Failed to resolve: %s', 'activitypub-auto-mention' ), esc_html( $mention ) )
                    );
            }

            // Set sanitised value
            $setting[ 'mentions' ] = join( "\n", $mentions );

            return $setting;
        }

    }

    AP_AutoMentions::instance();
} );
