<?php

namespace ReneSeindal;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly


/************************************************************************
 *
 *	Settings
 *
 ************************************************************************/

trait PluginBaseSettings {

    // Add settings link to plugin list
    function do_plugin_action_links___PLUGIN_BASENAME___filter_settings( $links ) {
        $url = $this->build_admin_url( 'options-general.php', [ 'page' => $this->option_page ] );
        $text = __( 'Settings', 'rs-base-plugin' );
        $links[] = sprintf( '<a href="%s">%s</a>', esc_url( $url ), esc_html( $text ) );
        return $links;
    }

    protected function get_settings_menu() {
        if ( ! empty( self::$option_menu ) )
            return self::$option_menu;

        self::$option_menu = 'options-general.php';

        $plugin_count = 0;
        foreach ( get_plugins() as $plugin )
            if ( !empty( $plugin['RequiresPlugins'] ) && str_contains( $plugin['RequiresPlugins'], 'rs-base-plugin' ) )
                $plugin_count += 1;

        if ( $plugin_count > 5 ) {
            // This avoids a spurious "RS Plugins" entry at the top of the menu
            self::$option_menu = $this->option_page;
            add_menu_page( 'RS Plugins', 'RS Plugins', 'manage_options', self::$option_menu, '', 'dashicons-admin-generic', 81 );
        }

        return self::$option_menu;
    }

    // Simplified wrapper for add_options_page -- just the basics
    function settings_add_menu( $page_title, $menu_title ) {
        $settings_menu = $this->get_settings_menu();

        add_submenu_page(
            $settings_menu,
            $page_title,
            $menu_title,
            'manage_options',
            $this->option_page,
            [ $this, 'settings_page_html' ]
        );
    }

    function do_admin_init_action_settings_define_sections_and_fields() {
        if ( method_exists( $this, 'settings_define_sections_and_fields' ) ) {
            register_setting(
                $this->option_page,
                $this->option_name,
                [
                    'sanitize_callback' => [ $this, 'settings_sanitize_option' ],
                ]
            );

            $this->settings_define_sections_and_fields();
        }
    }

    // Override in derived class for action sanitizing.
    function settings_sanitize_option( $input = NULl ) {
        return $input;
    }

    /* top level menu: callback functions */
    function settings_page_html() {
        // check user capabilities
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        // check if the user have submitted the settings
        // wordpress will add the "settings-updated" $_GET parameter to the url
        if ( isset( $_GET['settings-updated'] ) ) {
            add_settings_error( $this->option_page, 'settings_saved', __( 'Settings Saved', 'rs-base-plugin' ), 'updated' );
        }

        print( '<div class="wrap">' );
        printf( '<h1>%s</h1>', esc_html( get_admin_page_title()  ) );
        print( '<form action="options.php" method="post">' );

        settings_fields( $this->option_page );
        do_settings_sections( $this->option_page );
        submit_button( __( 'Save Settings', 'rs-base-plugin') );

        print( '</form>' );
        print( '</div>' );
    }

    // Helpers to create sections and fields

    function settings_add_section( $section, $title, $description = '', $renderer = 'settings_section_print_intro_html', $args = [] ) {
        $section = $this->option_base . '_' . $section;
        $args = array_merge([ 'intro' => $description ], $args );

        add_settings_section(
            $section,
            $title,
            [ $this, $renderer ],
            $this->option_page,
            $args
        );

        return $section;
    }

    function settings_add_field( $field, $section, $title, $renderer, $args = [] ) {
        $args = array_merge( [
            'field' => $field,
            'label_for' => $this->option_base . '_' . $field,
            'settings' => $this->option_name,
            'default' => $this->get_option( $field ),
        ], $args );


        if ( !method_exists( $this, $renderer) )
            $renderer = "settings_field_" . $renderer . "_html";
        if ( !method_exists( $this, $renderer) )
            $renderer = "settings_field_input_html";

        add_settings_field(
            $this->option_base . '_' . $field,
            $title,
            [ $this, $renderer ],
            $this->option_page,
            $section,
            $args
        );
    }



    /**
     * custom option and settings:
     * callback functions
     */


    function settings_section_print_intro_html( $args ) {
        if ( ! empty( $args['intro'] ) ) {
            $intros = is_array( $args['intro'] ) ? $args['intro'] : [ $args['intro'] ];
            foreach ( $intros as $intro )
                printf( "<p>%s</p>", esc_html( $intro ) );
        }
    }

    // Output a menu with or without multiple selection
    function settings_field_select_html( $args ) {
        $value = $this->get_option( $args['field'] ) ?? $args['default'];

        if ( isset( $args['multiple'] ) && $args['multiple'] ) {
            if ( isset( $args['size'] ) ) {
                $size = intval( $args['size'] );

                if ( $size <= 0 )
                    $size = count( $args['values'] );
            } else
                $size = 4;

            printf( '<select multiple size="%d" id="%s" name="%s[%s][]">' . PHP_EOL,
                    esc_attr( $size ),
                    esc_attr( $args['label_for'] ),
                    esc_attr( $args['settings'] ),
                    esc_attr( $args['field'] ),
            );

        } else {
            printf( '<select id="%s" name="%s[%s]">' . PHP_EOL,
                    esc_attr( $args['label_for'] ),
                    esc_attr( $args['settings'] ),
                    esc_attr( $args['field'] )
            );
        }

        foreach ( $args['values'] as $k => $v ) {
            printf( '<option value="%s" %s>%s</option>' . PHP_EOL,
                    esc_attr( $k ),
                    ( ( is_array( $value ) ? in_array( $k, $value ) : $k == $value ) ? 'selected' : '' ),
                    esc_html( $v )
            );
        }
        printf( "</select>" );

        if ( isset( $args['help'] ) )
            printf( '<div>%s</div>', $args['help'] );
    }

    // Output a text input field
    function settings_field_input_html( $args ) {
        $value = $this->get_option( $args['field'] ) ?? $args['default'];

        printf( '<input type="%s" size="%d id="%s" name="%s[%s]" value="%s">',
                esc_attr( $args['type'] ?? 'text' ),
                esc_attr( intval( $args['size'] ?? 20 ) ),
                esc_attr( $args['label_for'] ),
                esc_attr( $args['settings'] ),
                esc_attr( $args['field'] ),
                esc_attr( $value )
        );

        if ( isset( $args['validator'] ) and is_callable( $args['validator'] ) and !empty( $value ) and !empty( $args['error'] ) ) {
            $valid = call_user_func( $args['validator'], $value );
            if ( !$valid )
                printf( '<div style="color:red">%s</div>', $args['error'] );
        }
        if ( isset( $args['help'] ) )
            printf( '<div>%s</div>', $args['help'] );
    }

    // Output a checkbox
    function settings_field_checkbox_html( $args ) {
        $checked = $this->get_option( $args['field'] ) ?? $args['default'];

        if ( $checked )
            printf( '<input type="checkbox" id="%s" name="%s[%s]" value="%s" checked>',
                    esc_attr( $args['label_for'] ),
                    esc_attr( $args['settings'] ),
                    esc_attr( $args['field'] ),
                    esc_attr( $args['value'] ?? 'on' )
            );
        else
            printf( '<input type="checkbox" id="%s" name="%s[%s]" value="%s">',
                    esc_attr( $args['label_for'] ),
                    esc_attr( $args['settings'] ),
                    esc_attr( $args['field'] ),
                    esc_attr( $args['value'] ?? 'on' )
            );

        if ( isset( $args['help'] ) )
            printf( '<div>%s</div>', $args['help'] );
    }


    // Output a textarea
    function settings_field_textarea_html( $args ) {
        $value = $this->get_option( $args['field'] ) ?? $args['default'];

        printf( '<textarea id="%s" name="%s[%s]" cols="%d" rows="%d">%s</textarea>',
                esc_attr( $args['label_for'] ),
                esc_attr( $args['settings'] ),
                esc_attr( $args['field'] ),
                intval( $args['cols'] ?? 64 ),
                intval( $args['rows'] ?? 12 ),
                esc_attr( $value )
        );
        if ( isset( $args['help'] ) )
            printf( '<div>%s</div>', $args['help'] );
    }



    /************************************************************************
     *
     *	Helpers for post type multiple selection menus
     *
     ************************************************************************/

    // Add a settings field for a list of post types
    function settings_add_post_types_menu( $field, $section, $label, $post_types, $defaults = [] ) {
        $values = [];
        foreach ( $post_types as $post_type ) {
            $object = get_post_type_object( $post_type );
            if ( ! empty( $object ) )
                $values[ $object->name ] = $object->labels->name;
        }

        $this->settings_add_field(
            $field, $section, $label,
            'settings_field_select_html',
            [
                'default' => $defaults,
                'values' => $values,
                'multiple' => true,
                'size' => -1,
            ]
        );
    }

    // Make a list of post types, optionally filtering it
    function settings_prepare_post_types_menu( $query, $filters = NULL ) {
        $post_types = get_post_types( $query ?? [ 'public' => true ] );

        if ( $filters ) {
            if ( ! is_array( $filters ) )
                $filters = [ $filters ];

            foreach ( $filters as $filter )
                $post_types = apply_filters( $filter, $post_types );
        }

        return $post_types;
    }

    // Add a settings field with a filtered list of post types
    function settings_build_post_types_menu( $field, $section, $label, $query = NULL, $defaults = [], $filters = [] ) {
        $post_types = $this->settings_prepare_post_types_menu( $query, $filters );
        $this->settings_add_post_types_menu( $field, $section, $label, $post_types, $defaults );
    }

    /************************************************************************
     *
     *	Helpers for taxonomy multiple selection menus
     *
     ************************************************************************/

    // Add a settings field for a list of taxonomies
    function settings_add_taxonomies_menu( $field, $section, $label, $taxonomies, $defaults = [] ) {
        $values = [];
        foreach ( $taxonomies as $taxonomy ) {
            $object = get_taxonomy( $taxonomy );
            if ( ! empty( $object ) )
                $values[ $object->name ] = $object->labels->name;
        }

        $this->settings_add_field(
            $field, $section, $label,
            'settings_field_select_html',
            [
                'default' => $defaults,
                'values' => $values,
                'multiple' => true,
                'size' => -1,
            ]
        );
    }

    // Make a list of post types, optionally filtering it
    function settings_prepare_taxonomies_menu( $query, $filters = NULL ) {
        $taxonomies = get_taxonomies( $query ?? [ 'public' => true ] );

        if ( $filters ) {
            if ( ! is_array( $filters ) )
                $filters = [ $filters ];

            foreach ( $filters as $filter )
                $taxonomies = apply_filters( $filter, $taxonomies );
        }

        return $taxonomies;
    }

    // Add a settings field with a filtered list of post types
    function settings_build_taxonomies_menu( $field, $section, $label, $query = NULL, $defaults = [], $filters = [] ) {
        $taxonomies = $this->settings_prepare_taxonomies_menu( $query, $filters );
        $this->settings_add_taxonomies_menu( $field, $section, $label, $taxonomies, $defaults );
    }
}
