<?php

namespace ReneSeindal;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * RS Bibliography CLI commands
 *
 * ## EXAMPLES
 *
 *     wp rsbib export
 *
 * @when after_wp_load
 */

class BibliographyCLI {
    protected $plugin = NULL;

	function __construct() {
        $this->plugin = Bibliography::instance();
    }


    /**
     * Bibliography status
     *
     * ## EXAMPLES
     *
     *     wp rsbib status
     *
     */

    public function status( $args, $assoc ) {
        $terms = get_terms( [
            'taxonomy' => $this->plugin->get_taxonomy(),
            'hide_empty' => false,
        ] );

        $used = count( array_filter( $terms, fn($t) => $t->count > 0 ) );
        $unused = count( array_filter( $terms, fn($t) => $t->count == 0 ) );

        \WP_CLI::line( sprintf( 'Bibliography has %d entries (%d used, %d unused).', count( $terms ), $used, $unused ) );
    }


    /** $this->plugin->bibliography_fields()
     * Export bibliography entries
     *
     * ## EXAMPLES
     *
     *     wp rsbib export
     *
     */

    public function export( $args, $assoc ) {
        $terms = get_terms( [
            'taxonomy' => $this->plugin->get_taxonomy(),
            'hide_empty' => false,
        ] );

        $fields = array_keys(  $this->plugin->bibliography_fields() );

        $bib = [];

        foreach ( $terms as $term ) {
            $entry = [
                'name' => $term->name,
                'slug' => $term->slug,
                'description' => $term->description,
            ];

            foreach ( $fields as $field ) {
                $value = trim( get_term_meta( $term->term_id, $field, true ) );
                $entry[ substr( $field, 4 ) ] = $value;
            }

            $bib[] = $entry;
        }


        print json_encode( $bib, JSON_PRETTY_PRINT );
        print "\n";

    }

    /**
     * Import bibliography entries.
     *
     * ## OPTIONS
     *
     * <file>
     * : Read entries from <file>
     * ---
     *
     * ## EXAMPLES
     *
     *     wp rsbib import data.json
     *
     */

    public function import( $args, $assoc ) {
        $file = $args[0];

        if ( $file === '-' )
            $file = 'php://stdin';

        $json = @file_get_contents( $file );
        if ( $json === false )
            \WP_CLI::error( "Can't read file $file: " . (error_get_last())['message'] );

        if ( ! json_validate( $json ) )
            \WP_CLI::error( "Not valid JSON." );

        $data = json_decode( $json, 'array, please' );

        if ( ! is_array( $data ) )
            \WP_CLI::error( "Input not valid array." );

        // Key: field name ; value: whether term field or meta
        $fields = [
            "name" => false,
            "slug" => false,
            "description" => false,
            "author" => true,
            "title" => true,
            "serial" => true,
            "editor" => true,
            "year" => true,
            "link" => true,
        ];

        foreach ( $data as $entry ) {
            foreach ( array_keys( $fields ) as $field )
                if ( ! array_key_exists( $field, $entry ) )
                    \WP_CLI::error( "Input not complete." );

            if ( empty( $entry['slug'] ) )
                \WP_CLI::error( "Entry without slug." );
            if ( empty( $entry['name'] ) )
                \WP_CLI::error( "Entry without name." );
        }

        \WP_CLI::line( sprintf( 'Import data: %d entries.', count( $data ) ) );

        $taxonomy = $this->plugin->get_taxonomy();

        $existing = $terms = get_terms( [
            'taxonomy' => $taxonomy,
            'hide_empty' => false,
        ] );

        \WP_CLI::line( sprintf( 'Current data: %d entries.', count( $existing ) ) );

        $exmap = [];
        foreach ( $existing as $term )
            $exmap[$term->slug] = $term;

        $unchanged = 0;
        $updated = 0;
        $created = 0;

        foreach ( $data as $entry ) {
            if ( array_key_exists( $entry['slug'], $exmap ) ) {
                $term = $exmap[$entry['slug']];

                $changed = false;
                foreach ( $fields as $field => $meta ) {
                    if ( $meta )
                        $value = trim( get_term_meta( $term->term_id, "bib_$field", true ) );
                    else
                        $value = ( get_object_vars( $term ) )[$field];

                    if ( $value != $entry[$field] )
                        $changed = true;
                }

                if ( $changed ) {
                    \WP_CLI::line( sprintf( 'Updating %s.', $entry['slug'] ) );

                    wp_update_term( $term->term_id, $taxonomy, [
                        'name' => $entry['name'],
                        'description' => $entry['description'],
                    ] );

                    foreach ( $fields as $field => $meta ) {
                        if ( ! $meta ) continue;

                        $value = $entry[$field];

                        if ( empty( $value ) )
                            delete_term_meta( $term->term_id, "bib_$field" );
                        else
                            update_term_meta( $term->term_id, "bib_$field", $value );
                    }

                    $updated++;
                } else {
                    $unchanged++;
                }
            } else {
                \WP_CLI::line( sprintf( 'Creating %s.', $entry['slug'] ) );


                $res = wp_insert_term( $entry['name'], $taxonomy, [
                    'slug' => $entry['slug'],
                    'description' => $entry['description'],
                ] );

                if ( ! is_wp_error( $res ) ) {
                    $term_id = $res['term_id'];

                    foreach ( $fields as $field => $meta ) {
                        if ( ! $meta ) continue;

                        $value = $entry[$field];

                        if ( ! empty( $value ) )
                            add_term_meta( $term_id, "bib_$field", $value, true );
                    }
                }

                $created++;
            }
        }

        \WP_CLI::success( sprintf( 'created: %d -- updated: %d -- unchanged: %d', $created, $updated, $unchanged ) );
    }
}

\WP_CLI::add_command( 'rsbib', new BibliographyCLI() );
