<?php
/**
 * Plugin Name:         RS Bibliography
 * Description:         A really simple bibliography system with references stored in a taxonomy.
 * Plugin URI:          https://plugins.seindal.dk/plugins/rs-bibliography/
 * Update URI:          https://plugins.seindal.dk
 * Author:              René Seindal
 * Author URI:          https://plugins.seindal.dk/
 * Donate link:         https://mypos.com/@historywalks
 * Requires Plugins:    rs-base-plugin
 * License:             GPL v2 or later
 * License URI:         https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:         rs-bibliography
 * Domain Path:         /languages
 * Requires PHP:        7.4
 * Requires at least:   5.0
 * Version:             0.9.2
 **/

namespace ReneSeindal;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

require_once( ABSPATH . 'wp-admin/includes/taxonomy.php' );
require_once( ABSPATH . 'wp-admin/includes/class-wp-list-table.php' );

add_action( 'plugins_loaded', function() {
    class Bibliography extends PluginBase {
        protected $plugin_file = __FILE__;

        use PluginBaseSettings;

        protected $taxonomy = 'bibliography';

        /*************************************************************
         *
         * Registers the `bibliography` taxonomy
         *
         *************************************************************/

        function do_init_action_register_bibliography_taxonomy() {
            register_taxonomy( $this->taxonomy, [ 'post' ], [
                'hierarchical'          => false,
                'public'                => true,
                'show_in_nav_menus'     => true,
                'show_ui'               => true,
                'show_admin_column'     => true,
                'query_var'             => true,
                'rewrite'               => true,
                'capabilities'          => [
                    'manage_terms' => 'edit_posts',
                    'edit_terms'   => 'edit_posts',
                    'delete_terms' => 'edit_posts',
                    'assign_terms' => 'edit_posts',
                ],
                'labels'                => [
                    'name'                       => __( 'References', 'rs-bibliography' ),
                    'singular_name'              => _x( 'Reference', 'taxonomy general name', 'rs-bibliography' ),
                    'search_items'               => __( 'Search References', 'rs-bibliography' ),
                    'popular_items'              => __( 'Popular References', 'rs-bibliography' ),
                    'all_items'                  => __( 'All References', 'rs-bibliography' ),
                    'parent_item'                => __( 'Parent Reference', 'rs-bibliography' ),
                    'parent_item_colon'          => __( 'Parent Reference:', 'rs-bibliography' ),
                    'edit_item'                  => __( 'Edit Reference', 'rs-bibliography' ),
                    'update_item'                => __( 'Update Reference', 'rs-bibliography' ),
                    'view_item'                  => __( 'View Reference', 'rs-bibliography' ),
                    'add_new_item'               => __( 'Add New Reference', 'rs-bibliography' ),
                    'new_item_name'              => __( 'New Reference', 'rs-bibliography' ),
                    'separate_items_with_commas' => __( 'Separate references with commas', 'rs-bibliography' ),
                    'add_or_remove_items'        => __( 'Add or remove references', 'rs-bibliography' ),
                    'choose_from_most_used'      => __( 'Choose from the most used references', 'rs-bibliography' ),
                    'not_found'                  => __( 'No references found.', 'rs-bibliography' ),
                    'no_terms'                   => __( 'No references', 'rs-bibliography' ),
                    'menu_name'                  => __( 'References', 'rs-bibliography' ),
                    'items_list_navigation'      => __( 'References list navigation', 'rs-bibliography' ),
                    'items_list'                 => __( 'References list', 'rs-bibliography' ),
                    'most_used'                  => _x( 'Most Used', 'reference', 'rs-bibliography' ),
                    'back_to_items'              => __( '&larr; Back to References', 'rs-bibliography' ),
                ],
                'show_in_rest'          => true,
                'rest_base'             => $this->taxonomy,
                'rest_controller_class' => 'WP_REST_Terms_Controller',

                'update_count_callback' => '_update_generic_term_count',
            ] );

        }

        /*
         *
         * Sets the post updated messages for the `reference` taxonomy.
         *
         * @param  array $messages Post updated messages.
         * @return array Messages for the `reference` taxonomy.
         *
         */

        function do_term_updated_messages_filter_bibliography_updated_messages( $messages ) {

            $messages[$this->taxonomy] = [
                0 => '', // Unused. Messages start at index 1.
                1 => __( 'Reference added.', 'rs-bibliography' ),
                2 => __( 'Reference deleted.', 'rs-bibliography' ),
                3 => __( 'Reference updated.', 'rs-bibliography' ),
                4 => __( 'Reference not added.', 'rs-bibliography' ),
                5 => __( 'Reference not updated.', 'rs-bibliography' ),
                6 => __( 'References deleted.', 'rs-bibliography' ),
            ];

            return $messages;
        }


        /************************************************************
         *
         * Add filter on post lists on this taxonomy
         *
         ************************************************************/

        function do_restrict_manage_posts_action( $post_type, $which ) {
            if ( ! $this->get_option( 'post_list_filter', false ) )
                return;

            $this->restrict_manage_posts_taxonomy_menu( $post_type, $which, $this->taxonomy );
        }


        /************************************************************
         *
         * Add columms to bibliography table
         *
         ************************************************************/

        protected $manage_edit_bibliography_columns_callback_hook = 'manage_edit-bibliography_columns';

        function do_manage_edit_bibliography_columns_filter( $columns ) {
            $columns['bib_short'] = __( 'Short ref', 'rs-bibliography' );
            $columns['bib_full'] = __( 'Full ref', 'rs-bibliography' );
            return $columns;
        }

        function do_manage_bibliography_custom_column_action( $string, $column, $term_id ) {
            switch ( $column ) {
            case 'bib_short':
                echo $this->format_reference_short( $term_id );
                break;
            case 'bib_full':
                echo $this->format_reference_long( $term_id );
                break;
            }
        }


        /************************************************************
         *
         * Custom fields for bibliography entries
         *
         ************************************************************/

        public function bibliography_fields() {
            return [
                'bib_author' => [
                    'name' => __( 'Author', 'rs-bibliography' ),
                    'description' => __( 'Name of the author(s) -- separate multiple authors with //.', 'rs-bibliography' ),
                ],
                'bib_title' => [
                    'name' => __( 'Title', 'rs-bibliography' ),
                    'description' => __( 'The title of the work.', 'rs-bibliography' ),
                ],
                'bib_serial' => [
                    'name' => __( 'Serial / Collection', 'rs-bibliography' ),
                    'description' => __( 'The series, journal or collection the work is in.', 'rs-bibliography' ),
                ],
                'bib_editor' => [
                    'name' => __( 'Publisher', 'rs-bibliography' ),
                    'description' => __( 'The publisher and/or editor of the work.', 'rs-bibliography' ),
                ],
                'bib_year' => [
                    'name' => __( 'Year', 'rs-bibliography' ),
                    'description' => __( 'The publishing year of the work.', 'rs-bibliography' ),
                ],
                'bib_link' => [
                    'name' => __( 'Link', 'rs-bibliography' ),
                    'description' => __( 'Optional link for resource.', 'rs-bibliography' ),
                ],
            ];
        }

        function do_bibliography_add_form_fields_action() {
            $fields = $this->bibliography_fields();

            foreach ( $fields as $field => $labels ) {
                echo '<div class="form-field">';
                printf( '<label for="%s">%s</label>',
                        esc_attr( $field ), esc_html( $labels['name'] ) );
                printf( '<input name="%s" id="%s" type="text" value="">',
                        esc_attr( $field ), esc_attr( $field ) );
                printf( '<p id="%s-description">%s</p>',
                        esc_attr( $field ), esc_html( $labels['description'] ) );
                echo '</div>';
            }
        }

        function do_bibliography_edit_form_fields_action( $term ) {
            $fields = $this->bibliography_fields();

            foreach ( $fields as $field => $labels ) {
                $value = trim( get_term_meta( $term->term_id, $field, true ) );

                echo '<tr class="form-field">';
                printf( '<th scope="row"><label for="%s">%s</label></th>',
                        esc_attr( $field ), esc_html( $labels['name'] ));
                printf( '<td><input name="%s" id="%s" type="text" value="%s">',
                        esc_attr( $field ), esc_attr( $field ), esc_attr( $value ));
                printf( '<p class="description">%s</p></td>',
                        esc_html( $labels['description'] ));
                echo '</tr>';
            }
        }

        protected function store_term_field( $term_id, $query_var, $field = NULL ) {
            if ( !isset( $_REQUEST[$query_var] ) )
                return;

            $value = trim( $_REQUEST[$query_var] ?? '' );

            if ( !$field )
                $field = $query_var;

            if ( empty( $value ) )
                delete_term_meta( $term_id, $field );
            else
                update_term_meta( $term_id, $field, $value );
        }

        function do_create_bibliography_action( $term_id ) {
            $fields = $this->bibliography_fields();
            foreach ( $fields as $field => $labels )
                $this->store_term_field( $term_id, $field );
        }

        function do_edited_bibliography_action( $term_id ) {
            $this->do_create_bibliography_action( $term_id );
        }


        /************************************************************
         *
         * Bibliography entry formatting -- site dependent, but can be
         * overridden by filters.
         *
         * Defaults are:
         * Long form: AUTHOR. _TITLE_. EDITOR, YEAR.
         * Short form: AUTHOR (YEAR)
         *
         ************************************************************/

        protected function get_term_object( $term ) {
            $fields = $this->bibliography_fields();

            $data= [];
            foreach ( array_keys( $fields ) as $field ) {
                $pretty = $field;
                if ( str_starts_with( $field, 'bib_' ) )
                    $pretty = substr( $field, 4 );

                $data[$pretty] = trim( get_term_meta( $term->term_id, $field, true ) );
            }

            if ( !empty( $data['author'] ) ) {
                $authors = $this->parse_authors( $data['author'] );

                if ( count( $authors ) > 1 && str_starts_with( $authors[count( $authors )-1][0], '(' ) )
                    $data['eds'] = ( array_pop( $authors ) )[0];

                $data['authors'] = $authors;
            }

            return (object)$data;
        }

        // Parse author lists separated by // with optional (eds.) at the end
        public function parse_authors( $author ) {
            return array_map(
                fn( $a ) => array_map(
                    fn ( $a ) => trim( $a ),
                    explode( ',', $a, 2 )
                ),
                explode( '//', $author )
            );
        }

        // Generate short reference
        function format_reference_short( $term ) {
            $term = get_term( $term );
            return apply_filters( 'rs_bibliography_bibrefs', '', $this->get_term_object( $term ), $term );
        }

        // Default short reference
        function do_rs_bibliography_bibrefs_filter( $text, $ref, $term ) {
            if ( $text )
                return $text;

            if ( empty( $ref->author ) ) {
                if ( empty( $ref->title ) )
                    return esc_html( $term->name ); // fallback no author no title

                $title = $ref->title;
                if ( str_contains( $title, '//' ) )
                    list ( $title, $discard ) = array_map( fn( $s ) => trim( $s ),  explode( '//', $title ));

                list ($first, $last) = preg_split( '/[,.;:()-]/', $title . '.', 2 );
                $author =  sprintf( '<i>%s%s</i>', esc_html( $first ), ( empty( $last ) ? '' : '...' ) );
            } else {
                if ( count( $ref->authors ) > 1 )
                    $author = sprintf( '%s <i>et.al.</i>', esc_html( $ref->authors[0][0] ) );
                else
                    $author = esc_html( $ref->authors[0][0] );
            }

            return sprintf( '%s (%04d)', $author, $ref->year ?? __( 'n.d.', 'rs-bibliography' ) );
        }

        // Generate log reference
        function format_reference_long( $term ) {
            $term = get_term( $term );
            return apply_filters( 'rs_bibliography_bibrefl', '', $this->get_term_object( $term ), $term );
        }

        function do_rs_bibliography_bibrefl_filter( $text, $ref, $term ) {
            if ( $text )
                return $text;

            $periods = [];

            if ( !empty( $ref->author ) )
                $periods[] = esc_html( $this->format_author_full( $ref->authors, $ref->eds ?? NULL ) );

            if ( !empty( $ref->title ) ) {
                if ( empty( $ref->serial ) ) {
                    if ( str_contains( $ref->title, '//' ) ) {
                        list ( $title, $material ) = array_map( fn( $s ) => trim( $s ),  explode( '//', $ref->title ));
                        $periods[] = sprintf( '<i>%s</i>, %s', esc_html( $title ), esc_html( $material ) );
                    }  else
                        $periods[] = sprintf( '<i>%s</i>', esc_html( $ref->title ) );
                } else
                    $periods[] = sprintf( '<i>%s</i> in %s', esc_html( $ref->title ), esc_html( $ref->serial ) );
            }

            $wherewhen = [];
            if ( !empty( $ref->editor ) )
                $wherewhen[] = esc_html( $ref->editor );
            if ( !empty( $ref->year ) )
                $wherewhen[] = esc_html( $ref->year );

            if ( !empty( $wherewhen ) )
                $periods[] = join(', ', array_map( fn( $s ) => rtrim( $s, ',' ), $wherewhen ) );

            if ( empty( $periods ) )
                $periods[] = esc_html( $term->name );

            return join( '. ', array_map( fn( $s ) => rtrim( $s, '.' ), $periods ) );
        }

        // Format a full list of authors or editors
        public function format_author_full( $authors, $eds ) {
            // List of formatted names
            $names = [];

            // First author is always "LAST, FIRST"
            $first = array_shift( $authors );
            $names[] = join(', ', $first );

            // Subsequent authors are "FIRST LAST"
            foreach ( $authors as $a )
                $names[] = join(' ', array_reverse( $a ) );

            // Separate authors by commas, but 'and' before the last
            $last = array_pop( $names );
            if ( empty( $names ) )
                $namelist = $last;
            else {
                $namelist = join( ', ', $names ) . ' and ' . $last;
            }

            // Don't for editors
            if ( $eds )
                return $namelist . ' ' . $eds;

            return $namelist;
        }

        /************************************************************
         *
         * Bibliography shortcodes
         *
         ************************************************************/

        function format_bibliography( $terms, $header, $header_option, $html, $html_option, $classes, $mincount = 1 ) {
            if ( !is_array( $terms ) )
                return '';

            usort( $terms, fn($a,$b) => ( strtolower( $a->name ) <=> strtolower( $b->name ) ) );

            if ( false === $header )
                $header = $this->get_option( $header_option );

            if ( false === $html )
                $html = $this->get_option( $html_option );

            $header_output = NULL;
            if ( !empty( $header ) && !empty( $html ) )
                $header_output = sprintf( '<%s class="%s">%s</%s>',
                                          esc_attr( $html ),
                                          esc_attr( $classes ),
                                          esc_html( $header ),
                                          esc_attr( $html )
                );

            $output = [];
            $wrapper = NULL;
            $entry = $this->get_option( 'entry_tag'  );

            if ( ! in_array( $entry, [ 'p', 'div', 'ul', 'ol' ] ) )
                $entry = 'p';

            if ( in_array( $entry, [ 'ul', 'ol' ] ) ) {
                $wrapper = $entry;
                $entry = 'li';
            }

            foreach ( $terms as $term ) {
                $link = trim( get_term_meta( $term->term_id, 'bib_link', true ) );

                $text = $this->format_reference_long( $term ) . '.';

                if ( $term->count > $mincount )
                    $text .= sprintf( ' [<a href="%s">more</a>]', get_term_link( $term ) );

                if ( $link )
                    $text .= sprintf( ' <a href="%s">🔗</a>', esc_attr( $link ) );


                $output[] = sprintf( '<%s class="%s">%s</%s>', $entry, $classes, $text, $entry );
            }

            return ( $header_output ?: '')
                . ( $wrapper ? "<$wrapper>" : '' )
                . join( "\n", $output )
                . ( $wrapper ? "</$wrapper>" : '' );

        }

        // Shortcode [bibliography] outputs a bibliography based on tags used.

        function do_bibliography_shortcode( $atts = [], $content = null, $tag = '' ) {
            // normalize attribute keys, lowercase
            $atts = array_change_key_case( (array)$atts, CASE_LOWER );

            // override default attributes with user attributes
            $atts = shortcode_atts( [
                'header' => false,
                'html' => false,
            ] , $atts, $tag );

            $post = get_post();

            if ( empty( $post ) )
                return '';

            return $this->format_bibliography(
                get_the_terms( $post, $this->taxonomy ),
                $atts['header'], 'title',
                $atts['html'], 'title_tag',
                'bibliography'
            );
        }

        // Shortcode [full_bibliography] outputs everything in the
        // taxonomy with a non-zero count

        function do_full_bibliography_shortcode( $atts = [], $content = null, $tag = '' ) {
            // normalize attribute keys, lowercase
            $atts = array_change_key_case( (array)$atts, CASE_LOWER );

            // override default attributes with user attributes
            $atts = shortcode_atts( [
                'header' => false,
                'html' => false,
            ] , $atts, $tag );

            return $this->format_bibliography(
                get_terms( [ 'taxonomy' => $this->taxonomy] ),
                $atts['header'], 'fulltitle',
                $atts['html'], 'fulltitle_tag',
                'bibliography full-bibliography',
                0
            );
        }

        // Stupid hack to do manual bibliographies.

        private $default_link_term = false;
        private $default_link_external = false;

        // For ibid. and long first, then short refs.

        protected $last_ref = NULL;
        protected $seen_refs = [];

        function do_bibdefault_shortcode( $atts = [], $content = null, $tag = '' ) {
            // normalize attribute keys, lowercase
            $atts = array_change_key_case( (array)$atts, CASE_LOWER );

            // override default attributes with user attributes
            $atts = shortcode_atts( [
                'linkterm' => false,
                'linkext' => false,
            ] , $atts, $tag );

            $this->default_link_term = filter_var( $atts['linkterm'], FILTER_VALIDATE_BOOL );
            $this->default_link_external = filter_var( $atts['linkext'], FILTER_VALIDATE_BOOL );
            return '';
        }

        // Shortcodes [bibref* slug=XXX] or just [bibref* XXX] outputs a
        // reference to a work, whether tagged or not.

        function bibref_shortcode_wrapper( $atts = [], $content = null, $tag = '' ) {
            // normalize attribute keys, lowercase
            $atts = array_change_key_case( (array)$atts, CASE_LOWER );

            // override default attributes with user attributes
            $atts = shortcode_atts( [
                0 => NULL,          // XXX Hrmff
                'slug' => NULL,
                'linkterm' => false,
                'linkext' => false,
            ] , $atts, $tag );

            $slug = $atts['slug'] ?? $atts[0];

            if ( $slug )
                $term = get_term_by( 'slug', $slug, $this->taxonomy );

            if ( ! is_a( $term, 'WP_Term' ) )
                return sprintf( __( '[[BIBREF ERROR slug=%s]]', 'rs-bibliography' ), esc_html( $slug ) );

            $output = apply_filters( '_rs_bibliography_' . $tag, $term->name, $slug, $term );

            if ( filter_var( $atts['linkterm'], FILTER_VALIDATE_BOOL ) || $this->default_link_term )
                if ( $term->count > 0 )
                    $output .= sprintf( ' [<a href="%s">more</a>]', get_term_link( $term ) );

            if ( filter_var( $atts['linkext'], FILTER_VALIDATE_BOOL ) || $this->default_link_external ) {
                $link = trim( get_term_meta( $term->term_id, 'bib_link', true ) );
                if ( $link )
                    $output .= sprintf( ' <a href="%s">🔗</a>', esc_attr( $link ) );
            }

            $this->last_ref = $slug;

            return $output;
        }

        function do_bibrefs_shortcode( $atts = [], $content = null, $tag = '' ) {
            return $this->bibref_shortcode_wrapper( $atts, $content, $tag );
        }

        function do__rs_bibliography_bibrefs_filter( $output, $slug, $term ) {
            return $this->format_reference_short( $term );
        }

        function do_bibrefl_shortcode( $atts = [], $content = null, $tag = '' ) {
            return $this->bibref_shortcode_wrapper( $atts, $content, $tag );
        }

        function do__rs_bibliography_bibrefl_filter( $output, $slug, $term ) {
            return $this->format_reference_long( $term );
        }

        function do_bibref_shortcode( $atts = [], $content = null, $tag = '' ) {
            return $this->bibref_shortcode_wrapper( $atts, $content, $tag );
        }

        function do__rs_bibliography_bibref_filter( $output, $slug, $term ) {
            if ( $this->get_option( 'bibref_ibid' ) )
                if ( isset( $this->last_ref ) && $this->last_ref == $slug )
                    return __( '<i>ibid.</i>', 'rs-bibliography' );

            $this->last_ref = $slug;

            if ( $this->get_option( 'bibref_short_only' ) || array_key_exists( $slug, $this->seen_refs ) )
                return $this->format_reference_short( $term );

            $this->seen_refs[$slug] = true;

            return $this->format_reference_long( $term );
        }

        /*************************************************************
         *
         * Connect taxonomy and post types
         *
         *************************************************************/

        function do_wp_loaded_action() {
            $taxonomy = get_taxonomy( $this->taxonomy );
            $post_types = $this->get_option( 'post_types', $taxonomy->object_type );

            foreach ( $post_types as $post_type )
                if ( post_type_exists( $post_type ) )
                    register_taxonomy_for_object_type( $this->taxonomy, $post_type );
        }

        /************************************************************************
         *
         *	Management page - primarily for searching
         *
         ************************************************************************/

        public $management_page = NULL;
        public $management_page_name = 'rs-bibliography-page';

        function do_admin_menu_action_tools_page() {
            $this->management_page = add_management_page(
                __('References', 'rs-bibliography' ),
                __('RS Bibliography', 'rs-bibliography' ),
                'edit_posts',
                $this->management_page_name,
                [ $this, 'management_page_render' ]
            );

            add_action( "load-{$this->management_page}", [ $this, 'load_managament_page_hook' ] );
        }

        function management_page_link( $copy = true ) {
            $args = [ 'page' => $this->management_page_name ];

            if ( $copy ) {
                foreach ( [ 'order', 'orderby', 's' ] as $param ) {
                    if ( ! array_key_exists( $param, $args ) && isset( $_REQUEST[$param] ) )
                        $args[$param] = wp_unslash( $_REQUEST[$param] );
                }
            }

            return $this->build_admin_url( 'tools.php', $args );
        }

        function management_page_render() {
            printf( '<div class="wrap"><h1 class="wp-heading-inline">%s</h1>',
                    __( 'RS Bibliography', 'rs-bibliography' ) );
            printf( '<a href="%s" class="page-title-action">%s</a>',
                    $this->build_admin_url( 'edit-tags.php', [ 'taxonomy' => 'bibliography' ] ),
                    __( 'Add a New Reference', 'rs-bibliography' ) );
            printf( '<hr class="wp-header-end">' );

            printf( '<form method="get">' );
            printf( '<input type="hidden" name="page" value="%s" />', esc_attr( $this->management_page_name ) );

            // Used to copy [bibref] shortcode to clipboard
            echo '<script type="text/javascript">function rs_bibliography_copy_bibref( elem ) { navigator.clipboard.writeText(elem.innerText); return false; }</script>';

            $table = new RS_Bibliography_Table( $this, $this->management_page );

            $table->do_bulk_actions();

            $table->prepare_items();
            $table->views();
            $table->search_box( __( 'Search', 'rs-bibliography' ), $this->management_page . '-search' );
            $table->display();

            // Immediately focus search input
            $js = 'jQuery(document).ready(function($) { $( ".search-box #tools_page_rs-bibliography-search-search-input" ).each( function () { $(this).focus(); } ); } );';
            echo "<script type='text/javascript'>{$js}</script>";

            echo '</form>';
            echo '</div>';
        }

        // Set up the Screen Options

        function load_managament_page_hook() {
            $screen = get_current_screen();

            // get out of here if we are not on our settings page
            if( !is_object( $screen ) || $screen->id != $this->management_page )
                return;

            $args = array(
                'label' => __( 'Elements per page', 'rs-bibliography' ),
                'default' => 20,
                'option' => 'elements_per_page'
            );

            add_screen_option( 'per_page', $args );

            // This allows column selection - I have no idea why
            new RS_Bibliography_Table( $this, $this->management_page );
        }

        protected $set_screen_option_callback_hook = 'set-screen-option';

        function do_set_screen_option_filter( $status, $option, $value ) {
            return $value;
        }


        /************************************************************************
         *
         *	Dashboard search box for bibliography
         *
         ************************************************************************/

        function do_wp_dashboard_setup_action() {
            wp_add_dashboard_widget(
                "rs-bibliography-dashboard-widget",
                esc_html__( 'RS Bibliography', 'rs-dashboard-widgets' ),
                [ $this, 'dashboard_widget_render' ]
            );
        }

        function dashboard_widget_render() {
            $url =  admin_url( 'tools.php' );
            printf( '<form method="get" action="%s">', esc_url( $url ) );
            printf( '<input type="hidden" name="page" value="%s" />', esc_attr( $this->management_page_name ) );
            printf( '<div>%s</div>', esc_html__( 'Enter search terms:', 'rs-bibliography' ) );
            printf( '<input type="text" name="s" value="" />' );
            printf( '<input type="submit" id="unified-search-submit" class="button" value="Search">' );
            printf( '</form>' );
        }


        /************************************************************************
         *
         *	Settings
         *
         ************************************************************************/

        function do_admin_menu_action() {
            $this->settings_add_menu(
                __( 'Bibliography', 'rs-bibliography'),
                __( 'RS Bibliography', 'rs-bibliography' )
            );
        }

        // Called automatically by PluginBaseSettings
        function settings_define_sections_and_fields() {
            $section = $this->settings_add_section(
                'section_post_types',
                __( 'Post-types', 'rs-bibliography' ),
                __( 'Select the post-types you want bibliographies in.', 'rs-bibliography' )
            );

            $taxonomy = get_taxonomy( $this->taxonomy );

            $this->settings_build_post_types_menu(
                'post_types', $section,
                __( 'Post-types', 'rs-bibliography' ),
                [ 'public' => true ],
                $taxonomy->object_type,
                'rs_bibliography_post_types'
            );

            $section = $this->settings_add_section(
                'section_integration',
                __( 'Integrations', 'rs-bibliography' ),
                __( 'Select how this plugin integrates into the WordPress admin interface', 'rs-bibliography' )
            );

            $this->settings_add_field(
                'post_list_filter', $section,
                __( 'Allow post filtering', 'rs-bibliography' ),
                'settings_field_checkbox_html'
            );


            $section = $this->settings_add_section(
                'section_bibliography_entry',
                __( 'Bibliography entries', 'rs-bibliography' ),
                __( 'Formatting options for individual bibliography entries', 'rs-bibliography' )
            );

            $this->settings_add_field(
                'entry_tag', $section,
                __( 'HTML wrapper', 'rs-bibliography' ),
                'settings_field_select_html',
                [
                    'values' => [
                        'p' => __( 'Paragraph (<P>)', 'rs-bibliography' ),
                        'div' => __( 'Div (<DIV>)', 'rs-bibliography' ),
                        'ol' => __( 'Ordered list (<OL>)', 'rs-bibliography' ),
                        'ul' => __( 'Unordered list (<UL>)', 'rs-bibliography' ),
                    ],
                    'default' => 'p',
                ]
            );

            $headers = [
                '' => __( 'No automatic header', 'rs-bibliography' ),
                'h1' => __( 'Level 1 (<H1>)', 'rs-bibliography' ),
                'h2' => __( 'Level 2 (<H2>)', 'rs-bibliography' ),
                'h3' => __( 'Level 3 (<H3>)', 'rs-bibliography' ),
                'h4' => __( 'Level 4 (<H4>)', 'rs-bibliography' ),
                'h5' => __( 'Level 5 (<H5>)', 'rs-bibliography' ),
                'h6' => __( 'Level 6 (<H6>)', 'rs-bibliography' ),
                'div' => __( '<DIV class="bibliography">', 'rs-bibliography' ),
            ];

            $section = $this->settings_add_section(
                'section_post_bibliography',
                __( 'Single post bibliography', 'rs-bibliography' ),
                __( 'Heading output by [bibliography] shortcode.', 'rs-bibliography' )
            );

            $this->settings_add_field(
                'title', $section,
                __( 'Heading', 'rs-bibliography' ),
                'settings_field_input_html',
                [
                    'default' => __( 'Bibliography', 'rs-bibliography' ),
                    'size' => 40,
                ]
            );

            $this->settings_add_field(
                'title_tag', $section,
                __( 'HTML wrapper', 'rs-bibliography' ),
                'settings_field_select_html',
                [
                    'values' => $headers,
                    'default' => 'h2',
                ]
            );

            $section = $this->settings_add_section(
                'section_full_bibliography',
                __( 'Full bibliography', 'rs-bibliography' ),
                __( 'Heading output by [full_bibliography] shortcode.', 'rs-bibliography' )
            );

            $this->settings_add_field(
                'fulltitle', $section,
                __( 'Heading', 'rs-bibliography' ),
                'settings_field_input_html',
                [
                    'default' => __( 'Site bibliography', 'rs-bibliography' ),
                    'size' => 40,
                ]
            );

            $this->settings_add_field(
                'fulltitle_tag', $section,
                __( 'HTML wrapper', 'rs-bibliography' ),
                'settings_field_select_html',
                [
                    'values' => $headers,
                    'default' => 'h2',
                ]
            );

            $section = $this->settings_add_section(
                'section_bibref',
                __( 'The [bibref] shortcode', 'rs-bibliography' ),
                [
                    __( 'Details of the outpug of the [bibref] shortcode.', 'rs-bibliography' ),
                    __( 'Default behaviour is to use the long form the first time a work is referenced, then the short form, but for consecutive references to the same work, replace the short form with "ibid."', 'rs-bibliography' ),
                ]
            );

            $this->settings_add_field(
                'bibref_short_only', $section,
                __( 'Always use the short form', 'rs-bibliography' ),
                'settings_field_checkbox_html'
            );

            $this->settings_add_field(
                'bibref_ibid', $section,
                __( 'Use "ibid."', 'rs-bibliography' ),
                'settings_field_checkbox_html'
            );
        }



        /************************************************************************
         *
         *	CLI helpers
         *
         ************************************************************************/

        public function get_taxonomy() {
            return $this->taxonomy;
        }
    }

    Bibliography::instance();

    /************************************************************************
     *
     *	Bibliography Table
     *
     ************************************************************************/

    class RS_Bibliography_Table extends \WP_List_Table {
        protected $plugin = NULL;
        protected $table_data;  // Display data

        function __construct( $plugin ) {
            parent::__construct();

            $this->plugin = $plugin;
            $this->table_data = [];
        }

        private function build_admin_url( $page, $args ) {
            return add_query_arg( $args, admin_url( $page ) );
        }

        private function req_var( $name, $default = NULL ) {
            return ( !empty( $_REQUEST[$name] ) ? sanitize_key( wp_unslash( $_REQUEST[$name] ) ) : $default );
        }

        function get_columns() {
            $columns = [
                'cb' => '<input type="checkbox" />',
                'author' => __('Author', 'rs-bibliography'),
                'title' => __('Title', 'rs-bibliography'),
                'serial' => __('Serial', 'rs-bibliography'),
                'publisher' => __('Publisher', 'rs-bibliography'),
                'year' => __('Year', 'rs-bibliography'),
                'bibref' => __('Shortcode', 'rs-bibliography'),
                'shortref' => __('Short reference', 'rs-bibliography'),
                'longref' => __('Long reference', 'rs-bibliography'),
            ];

            return $columns;
        }

        // Bind table with columns, data and all
        function prepare_items() {
            // Get table data

            $args = [
                'taxonomy' => 'bibliography',
                'hide_empty' => false,
            ];

            $s = trim( $this->req_var( 's', '' ) );
            if ( $s ) {
                $args['meta_query'] = [
                    'relation' => 'OR',
                    [
                        'key' => 'bib_author',
                        'value' => $s,
                        'compare' => 'LIKE',
                    ],
                    [
                        'key' => 'bib_title',
                        'value' => $s,
                        'compare' => 'LIKE',
                    ],
                    [
                        'key' => 'bib_year',
                        'value' => $s,
                        'compare' => 'LIKE',
                    ],
                ];
            }

            $terms = get_terms( $args );

            $this->table_data = [];
            foreach ( $terms as $term ) {
                $this->table_data[] = [
                    'id' => $term->term_id,
                    'author' => trim( get_term_meta( $term->term_id, 'bib_author', true) ),
                    'title' => trim( get_term_meta( $term->term_id, 'bib_title', true) ),
                    'serial' => trim( get_term_meta( $term->term_id, 'bib_serial', true) ),
                    'publisher' => trim( get_term_meta( $term->term_id, 'bib_editor', true) ),
                    'year' => trim( get_term_meta( $term->term_id, 'bib_year', true) ),
                    'link' => trim( get_term_meta( $term->term_id, 'bib_link', true) ),
                    'bibref' => $term->slug,
                    'shortref' => $this->plugin->format_reference_short( $term ),
                    'longref' => $this->plugin->format_reference_long( $term ),
                ];
            }

            $this->sort_table_rows();

            // Do the columns and stuff
            $columns = $this->get_columns();
            $usermeta = get_user_meta( get_current_user_id(), "manage{$this->plugin->management_page}columnshidden", true);
            $hidden = ( is_array( $usermeta ) ? $usermeta : [] );
            $sortable = $this->get_sortable_columns();
            $primary  = 'author';

            $this->_column_headers = [ $columns, $hidden, $sortable, $primary ];

            // Pagination
            $per_page = $this->get_items_per_page( 'elements_per_page', 10);
            $current_page = $this->get_pagenum();
            $total_items = count($this->table_data);

            $this->table_data = array_slice( $this->table_data, ( ($current_page - 1) * $per_page ), $per_page );

            $this->set_pagination_args( [
                'total_items' => $total_items, // total number of items
                'per_page'    => $per_page, // items to show on a page
                'total_pages' => ceil( $total_items / $per_page ) // use ceil to round up
            ] );

            $this->items = $this->table_data;
        }

        // Default set value for each column
        function column_default( $item, $column_name ) {
            return $item[$column_name] ?: '';
        }

        // Add a checkbox in the first column
        function column_cb( $item ) {
            return sprintf( '<input type="checkbox" name="ids[]" value="%d" />',  $item['id'] );
        }

        function column_author( $item ) {
            $wp_http_referer = $this->plugin->management_page_link();

            $edit_url = $this->build_admin_url( 'term.php', [
                'taxonomy' => 'bibliography',
                'post_type' => 'post',
                'tag_ID' => $item['id'],
                'wp_http_referer' => str_replace( site_url(), '', $wp_http_referer ),
            ] );

            $actions['edit'] = sprintf( '<a href="%s" class="%s">%s</a>',
                                        esc_url( $edit_url ),
                                        esc_attr( 'rs-bib' ),
                                        esc_html( __( 'Edit', 'rs-bibliography' ) ) );

            $term = get_term( $item['id'] );
            if ( $term->count > 0 )
                $actions['view'] = sprintf( '<a href="%s" class="%s">%s</a>',
                                            esc_url( get_term_link( $item['id'] ) ),
                                            esc_attr( 'rs-bib' ),
                                            esc_html( __( 'View', 'rs-bibliography' ) ) );

            if ( empty( $item['author'] ) ) {
                return sprintf( '%s %s',
                                esc_html( __( 'No Author', 'rs-bibliography' ) ),
                                $this->row_actions( $actions )
                );
            }

            return sprintf( '<a href="%s">%s</a> %s',
                            esc_url( $edit_url ),
                            esc_html( $item['author'] ?: __( 'No Author', 'rs-bibliography' ) ),
                            $this->row_actions( $actions )
            );
        }

        // Default set value for each column
        function column_shortref( $item ) {
            return ( $item['link']
                     ? sprintf( '<a href="%s">%s</a>', esc_attr( $item['link'] ), esc_html( $item['shortref'] ) )
                     : $item['shortref']
            );
        }

        function column_bibref( $item ) {
            return sprintf( '<a title="%s" onclick="rs_bibliography_copy_bibref(this);">[bibref %s].</a>',
                            __( 'Click to copy shortcode to clipboard.', 'rs-bibliography' ),
                            $item['bibref'] );
        }

        protected function get_sortable_columns() {
            $sortable_columns = [
                'author'  => [ 'author', false ],
                'title'  => [ 'title', false ],
                'year'  => [ 'year', false ],
            ];
            return $sortable_columns;
        }

        protected function sort_table_rows() {
            $orderby = $this->req_var( 'orderby', 'author');
            $order = $this->req_var( 'order', 'asc');

            if ( $order == 'asc' )
                $cmp = fn($a, $b) => strcmp( $a[$orderby], $b[$orderby] );
            else
                $cmp = fn($a, $b) => strcmp( $b[$orderby], $a[$orderby] );

            usort( $this->table_data, $cmp );
        }

        function get_bulk_actions() {
            $bulk_actions = [];

            $actions = [];
            foreach ( $bulk_actions as $action => $data )
                $actions[$action] = $data['label'];

            return $actions;
        }

        function do_bulk_actions() {
            $action = $this->current_action();
            if ( $action )
                do_action( "rs_bibliography_page_bulk_{$action}" );
        }
    }

    if ( defined( 'WP_CLI' ) && WP_CLI ) {
        require_once dirname( __FILE__ ) . '/rs-bibliography-cli.php';
    }
} );
