<?php

namespace ReneSeindal;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/************************************************************************
 *
 *	Lists of search results of mixed post types
 *
 ************************************************************************/

require_once( ABSPATH . 'wp-admin/includes/class-wp-list-table.php' );

class Dashboard_Search_Post_Table extends \WP_List_Table {
    protected $page;
    protected $post_types;

    protected $table_data;
    protected $search_term;

    function __construct( $page, $post_types ) {
        parent::__construct();
        $this->page = $page;
        $this->post_types = $post_types;
    }

    function build_admin_url( $page, $args ) {
        return add_query_arg( $args, admin_url( $page ) );
    }

    private function req_var_raw( $name, $default = NULL ) {
        return ( !empty( $_REQUEST[$name] ) ? wp_unslash( $_REQUEST[$name] ) : $default );
    }

    private function req_var( $name, $default = NULL ) {
        return sanitize_key( $this->req_var_raw( $name, $default) );
    }

    private function get_post_stati( $what = 'names' ) {
        return get_post_stati( [ 'exclude_from_search' => false, 'internal' => false, ], $what );
    }

    function get_columns() {
        $columns = [
            'cb' => '<input type="checkbox" />',
            'title' => __('Title', 'rs-dashboard-widgets'),
            'type' => __('Type', 'rs-dashboard-widgets'),
            'status' => __('Status', 'rs-dashboard-widgets'),
            'date' => __('Date', 'rs-dashboard-widgets'),
            'match' => __('Match', 'rs-dashboard-widgets'),
        ];

        // XXX Not entirely correct but not entirely wrong either
        $columns = apply_filters( 'manage_posts_columns', $columns, 'post' );

        return $columns;
    }

    function add_items( $ids ) {
        $this->table_data = [];

        foreach ( $ids as $id ) {
            $post = get_post( $id );

            $this->table_data[] = [
                'id' => $post->ID,
                'title' => $post->post_title,
                'type' => $post->post_type,
                'status' => $post->post_status,
                'date' => $post->post_date,
                'match' => '',
            ];
        }

        return $this->table_data;
    }

    // Call after prepare_items() to save the search term
    function output_search_term() {
        if ( $this->search_term )
            printf( '<input type="hidden" name="s" value="%s">', esc_attr( $this->search_term ) );
    }

    protected function get_views() {
        $statuses = $this->get_post_stati();
        $post_types = $this->post_types;

        $links = [];
        foreach ( $post_types as $post_type ) {
            $ids = get_posts( [
                'post_type' => $post_type->name,
                'post_status' => $statuses,
                's' => $this->search_term,
                'fields' => 'ids',
                'numberposts' => -1,
            ] );

            if ( count( $ids ) ) {
                $url = $this->build_admin_url( 'edit.php', [ 'post_type' => $post_type->name, 's' => $this->search_term ] );
                $links[ $post_type->name ] = [
                    'url' => $url,
                    'label' => sprintf( '%s <span class="count">(%d)</span>', $post_type->labels->name, count( $ids ) ),
                ];
            }
        }

        return $this->get_views_links( $links );
    }

    function extra_tablenav( $which ) {
        if ( 'top' !== $which ) return;

        $statuses = $this->get_post_stati( 'objects' );
        $post_status = $this->req_var( 'post_status', '' );

        echo '<div class="alignleft actions">';
        echo '<select name="post_status">';
        printf( '<option value="any">%s</option>', __( 'Any status', 'rs-dashboard-widgets' ) );
        foreach ( $statuses as $status ) {
            printf( '<option value="%s" %s>%s</option>',
                    esc_attr( $status->name ),
                    ( $post_status == $status->name ) ? 'selected' : '',
                    esc_html( $status->label )
            );
        }
        echo '</select>';

        // XXX Not entirely correct but not entirely wrong either
        do_action( 'restrict_manage_posts', 'post', $which );

        submit_button( __('Filter', 'rs-dashboard-widgets' ), '', 'post_status_change', false );
        echo '</div>';

    }

    // Bind table with columns, data and all
    function prepare_items() {
        // Bulk actions
        $action = $this->current_action();
        if ( $action ) {
            $ids = $this->req_var( 'ids' );
            if ( !empty( $ids ) )
                $this->handle_bulk_actions( $action, $ids );
        }

        // Do the initial search

        $s = sanitize_text_field( $this->req_var_raw( 's' ) );
        if ( $s )
            $this->search_term = $s;

        $order = $this->req_var( 'order', 'desc' );
        $orderby = $this->req_var( 'orderby', 'relevance' );

        $post_status = $this->req_var( 'post_status' );
        $statuses = $this->get_post_stati();

        $tax_query = [];
        foreach ( get_taxonomies() as $taxonomy ) {
            $term = $this->req_var( $taxonomy );
            if ( $term ) {
                $tax_query[] = [
                    'taxonomy' => $taxonomy,
                    'field' => 'slug',
                    'terms' => $term,
                ];
            }
        }

        $query = [
            'post_type' => array_map( fn( $pto ) => $pto->name,  $this->post_types ),
            'post_status' => $post_status ?: $statuses,
            's' => $s,
            'tax_query' => $tax_query,
            'fields' => 'ids',
            'order' => $order,
            'orderby' => $orderby,
            'numberposts' => -1,
        ];

        $ids = get_posts( $query );

        // Table setup
        $columns = $this->get_columns();
        $usermeta = get_user_meta( get_current_user_id(), "manage{$this->page}columnshidden", true);
        $hidden = ( is_array( $usermeta ) ? $usermeta : [] );

        if ( empty( $s ) )
            $hidden[] = 'match';

        $sortable = $this->get_sortable_columns();
        $primary  = 'title';
        $this->_column_headers = [ $columns, $hidden, $sortable, $primary ];

        // Pagination
        $per_page = $this->get_items_per_page( 'elements_per_page', 10);
        $current_page = $this->get_pagenum();
        $total_items = count($ids);

        $this->set_pagination_args( [
            'total_items' => $total_items, // total number of items
            'per_page'    => $per_page, // items to show on a page
            'total_pages' => ceil( $total_items / $per_page ) // use ceil to round up
        ] );

        // Get the posts for the page
        $ids = array_slice( $ids, ( ($current_page - 1) * $per_page ), $per_page );

        $this->items = $this->add_items($ids);
    }

    // Default set value for each column
    function column_default( $item, $column_name )  {
        if ( isset( $item[$column_name] ) )
            echo $item[$column_name];
        else
            // XXX Not entirely correct but not entirely wrong either
            do_action( 'manage_posts_custom_column', $column_name, $item['id'] );

    }

    // Add a checkbox in the first column
    function column_cb( $item )  {
        return sprintf( '<input type="checkbox" name="ids[]" value="%d" />',  esc_attr( $item['id'] ) );
    }

    // Adding action links to title column
    function column_title( $item ) {
        $edit_url = $this->build_admin_url( 'post.php', [ 'post' => $item['id'], 'action' => 'edit' ] );
        $trash_url = $this->build_admin_url( 'post.php', [ 'post' => $item['id'], 'action' => 'trash', '_wpnonce' => wp_create_nonce( 'trash-post_' . $item['id'] ) ] );

        $actions = [
            'edit' => sprintf( '<a href="%s">%s</a>', esc_url( $edit_url ),
                               esc_html__( 'Edit', 'rs-dashboard-widgets' ) ),
            'trash' => sprintf( '<a href="%s">%s</a>', esc_url( $trash_url ),
                                esc_html__( 'Bin', 'rs-dashboard-widgets' ) )
        ];

        if ( is_post_type_viewable( $item['type'] ) ) {
            $status = get_post_status_object( $item['status'] );

            if ( $status->public ) {
                $view_url = get_permalink( $item['id'] );
                $view_label = __( 'View', 'rs-dashboard-widgets' );
            } else {
                $view_url = add_query_arg( [
                    'post_type' => $item['type'],
                    'p' => $item['id'],
                    'preview' => 'true',
                ], site_url() );
                $view_label = __( 'Preview', 'rs-dashboard-widgets' );
            }
            $actions['view'] = sprintf( '<a href="%s">%s</a>', esc_url( $view_url ), esc_html( $view_label ) );
        }

        $actions = apply_filters( 'post_row_actions', $actions, get_post( $item['id'] ) );

        return sprintf('<strong><a href="%s">%s</a></strong> %s',
                       esc_url( $edit_url ), esc_html( $item['title'] ),
                       $this->row_actions( $actions )
        );
    }

    function column_date( $item ) {
        if ( 'draft' === $item['status'] ) {
            $status_msg = __( 'Last Modified', 'rs-dashboard-widgets' );
            $datetime = get_post_datetime( $item['id'], 'modified' );
        } else {
            $status = get_post_status_object( $item['status'] );
            $status_msg = $status->label;
            $datetime = get_post_datetime( $item['id'] );
        }

        $date = sprintf( __( '%s at %s', 'rs-dashboard-widgets' ),
                         $datetime->format( 'Y/m/d' ),
                         $datetime->format( 'G:i' )
        );

        return $status_msg . '<br/>' . $date;
    }

    function column_status( $item ) {
        $obj = get_post_status_object( $item['status'] );
        return $obj->label;
    }

    function column_type( $item ) {
        $obj = get_post_type_object( $item['type'] );
        return $obj->label;
    }

    function column_match( $item ) {
        $s = trim( $this->search_term ?: '' );
        if ( empty( $s ) ) return '';

        // Quoted search works in WP, but we have to remove the quotes to find matches
        $s = trim( $s, '"\'');

        $post = get_post( $item['id'] );

        $found = false;
        $regex = '/(\W)?((?:\w+\W+){0,12}\w*?)(' . preg_quote( $s ) . ')((?:\w+)?(?:\W+\w+){0,12})(.)?/i';

        if ( preg_match($regex, wp_strip_all_tags( $post->post_content, true ), $m ) ) {
            $found = true;
        } else if ( preg_match($regex, $post->post_content, $m ) ) {
            $found = true;
        } else if ( preg_match($regex, $post->post_excerpt, $m ) ) {
            $found = true;
        } else if ( preg_match($regex, $post->post_title, $m ) ) {
            $found = true;
        } else if ( str_contains( $post->post_content, '<!-- wp:footnotes /-->' ) && !empty( $post->footnotes ) ) {
            try {
                $footnotes = wp_list_pluck( json_decode( $post->footnotes, true ), 'content' );
            }
            catch ( ValueException ) {
                $footnotes = NULL;
            }

            if ( !empty( $footnotes ) ) {
                foreach ( $footnotes as $footnote ) {
                    if ( $footnote ) {
                        if ( preg_match($regex, wp_strip_all_tags( $footnote ), $m ) ) {
                            $found = true;
                        } else if ( preg_match($regex, $footnote, $m ) ) {
                            $found = true;
                        }
                    }
                    if ( $found ) break;
                }
            }
        }

        if ( $found )
            return ( $m[1] ? '...' : '' )
                . esc_html( $m[2] )
                . '<span style="color: red; text-weight: bold; font-size: 120%;">' . esc_html( $m[3] ) . '</span>'
                . esc_html( $m[4] )
                . ( isset( $m[5] ) ? '...' : '' );

        return sprintf( __( 'No visible match of <strong>%s</strong> found.' ), esc_html( $s ) );
    }

    // Define sortable column
    protected function get_sortable_columns() {
        $sortable_columns = [
            'title'		=> [ 'post_title', true ],
            'type'		=> [ 'post_type', true ],
            'status'	=> [ 'post_status', true ],
            'date'		=> [ 'post_date', true ],
        ];

        // XXX Not entirely correct but not entirely wrong either
        $extras = apply_filters( 'manage_edit-post_sortable_columns', [] );
        foreach ( $extras as $extra )
            $sortable_columns[$extra] = [ $extra, true ];

        return $sortable_columns;
    }

    // To show bulk action dropdown
    function get_bulk_actions() {
        $actions = [
            'trash' => __( 'Bin', 'rs-dashboard-widgets' ),
        ];
        return $actions;
    }

    // Handle the bulk actions
    function handle_bulk_actions( $action, $post_ids ) {
        return;             // NOTIMPL TODO
    }
}
