<?php
/**
 * Plugin Name:         RS Dashboard Widgets
 * Description:         Updates and creates dashboard widgets for custom post types, and creates a unified search interface across all post types.
 * Plugin URI:          https://plugins.seindal.dk/plugins/rs-dashboard-widgets/
 * Update URI:          https://plugins.seindal.dk
 * Author:              René Seindal
 * Author URI:          https://plugins.seindal.dk/
 * Donate link:         https://mypos.com/@historywalks
 * Requires Plugins:    rs-base-plugin
 * License:             GPL v2 or later
 * License URI:         https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:         rs-dashboard-widgets
 * Domain Path:         /languages
 * Requires PHP:        7.4
 * Requires at least:   5.0
 * Version:             1.25.6
 **/

namespace ReneSeindal;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

include( __DIR__ . '/class-dashicons.php' );
include( __DIR__ . '/class-search-table.php' );

add_action( 'plugins_loaded', function() {
    class DashboardWidgets extends PluginBase {
        protected $plugin_file = __FILE__;

        use PluginBaseSettings;

        // Never handle these
        protected $post_type_blacklist = [ 'attachment' ];


        /*************************************************************
         *
         * Get post type lists from options
         *
         *************************************************************/

        function eliminate_blacklisted_post_types( $post_types ) {
            return array_filter(
                array_map(
                    fn( $name ) => is_a( $name, 'WP_Post_type' ) ? $name : get_post_type_object( $name ),
                    $post_types ?? []
                ),
                fn( $type ) =>  ( isset( $type )
                                  && ! in_array( $type->name, $this->post_type_blacklist )
                )
            );
        }

        function get_post_types_for_std_widgets() {
            $post_type_names = $this->get_option( 'post_types_std_widgets', [] );
            $post_types = $this->eliminate_blacklisted_post_types( $post_type_names );
            return array_filter( $post_types, fn( $pt ) => ! $pt->_builtin );
        }

        function get_post_types_for_new_widgets() {
            $post_type_names = $this->get_option( 'post_types_new_widgets', [] );
            return $this->eliminate_blacklisted_post_types( $post_type_names );
        }

        function get_post_types_for_unified_search() {
            $post_types = $this->get_option( 'post_types_search', ['post','page'] );
            return $this->eliminate_blacklisted_post_types( $post_types );
        }

        function get_post_types_for_rss_feed() {
            $post_type_names = $this->get_option( 'post_types_rss_feed', [ 'post' ] );
            $post_types = $this->eliminate_blacklisted_post_types( $post_type_names );
            return array_filter( $post_types, fn( $pt ) => is_post_type_viewable( $pt ) );
        }

        /*************************************************************
         *
         * At a glance - add new post-type counts with icons
         *
         *************************************************************/

        function do_dashboard_glance_items_filter( $items ) {
            $output = array_map(
                function( $post_type ) {
                    $name = $post_type->name;
                    $num_posts = wp_count_posts( $name );
                    $num = apply_filters( 'rs_dashboard_widgets_post_count', $num_posts->publish, $name, $num_posts );
                    $text = _n( $post_type->labels->singular_name,
                                $post_type->labels->name,
                                intval( $num )
                    );

                    return sprintf( '<a class="%s" href="edit.php?post_type=%s">%s %s</a>',
                                    "at-a-glance-{$name}-count",
                                    esc_attr( $name ),
                                    esc_html( number_format_i18n( $num ) ),
                                    esc_html( $text )
                    );
                }, $this->get_post_types_for_std_widgets()
            );

            return array_merge( $output, $items);
        }

        // Add the correct icons to the At a Glance dashboard widget
        function do_admin_head_action_at_a_glance_css() {

            $styles = [];
            foreach ( $this->get_post_types_for_std_widgets() as $post_type ) {
                $icon = $post_type->menu_icon;
                if ( $icon ) {
                    $selector = sprintf( '#dashboard_right_now li a.at-a-glance-%s-count::before',
                                         esc_attr( $post_type->name ) );

                    $properties = NULL;
                    if ( str_starts_with( $icon, 'data:' ) )
                        $properties =  sprintf( 'height: 20px; width: 20px; content: url(%s);', $icon );
                    else {
                        $codepoint = DashIcons::lookup( $icon );
                        if ( $codepoint  )
                            $properties = sprintf( 'content: "\%x";', $codepoint );
                    }

                    if ( $properties )
                        $styles[] = "$selector { $properties }";
                }
            }

            if ( $styles )
                echo '<style type="text/css">', join(PHP_EOL, $styles ), '</style>';
        }


        /*************************************************************
         *
         * Activity - add new post-types
         *
         *************************************************************/

        function do_dashboard_recent_posts_query_args_filter( $query_args ) {
            $post_types = $this->get_post_types_for_std_widgets();

            if ( ! is_array( $query_args['post_type'] ) )
                $query_args['post_type'] = [ $query_args['post_type'] ];

            $query_args['post_type'] = array_merge( $query_args['post_type'], wp_list_pluck( $post_types, 'name' ) );

            return $query_args;
        }

        /*************************************************************
         *
         * Recent drafts - add new post-types
         *
         *************************************************************/

        function do_dashboard_recent_drafts_query_args_filter( $query_args ) {
            return $this->do_dashboard_recent_posts_query_args_filter( $query_args );
        }


        /*************************************************************
         *
         * Post type status dashboard widgets
         *
         *************************************************************/

        function do_wp_dashboard_setup_action_post_type_status_widgets() {
            $post_types = $this->get_post_types_for_new_widgets();

            foreach ( $post_types as $post_type ) {
                wp_add_dashboard_widget(
                    "post-type-{$post_type->name}-dashboard-widget",
                    sprintf( esc_html__( 'Status: %s', 'rs-dashboard-widgets' ), $post_type->labels->all_items ),
                    function () use ( $post_type ) { $this->dashboard_widget_render( $post_type ); }
                );}

        }

        /**
         * Create the function to output the content of our Dashboard Widget.
         */
        function dashboard_widget_render( $post_type ) {
            printf( '<div class="post-type-status-widget" id="post-type-status-widget-%s">', esc_attr( $post_type->name ) );

            $this->dashboard_posts_status( $post_type );

            $title = sprintf( __( 'Scheduled %s', 'rs-dashboard-widgets' ), strtolower( $post_type->labels->name ) );
            $this->dashboard_posts_table( $post_type, 'future', $title, 'ASC' );

            $title = sprintf( __( 'Recently published %s', 'rs-dashboard-widgets' ), strtolower( $post_type->labels->name ) );
            $this->dashboard_posts_table( $post_type, 'publish', $title );

            $title = sprintf( __( 'Pending %s', 'rs-dashboard-widgets' ), strtolower( $post_type->labels->name ) );
            $this->dashboard_posts_table( $post_type, 'pending', $title, 'DESC', -1 );

            $title = sprintf( __( 'Recent %s drafts', 'rs-dashboard-widgets' ), strtolower( $post_type->labels->singular_name ) );
            $drafts = $this->dashboard_posts_table( $post_type, 'draft', $title, 'DESC', 5 );

            $title = sprintf( __( 'Old %s drafts', 'rs-dashboard-widgets' ), strtolower( $post_type->labels->singular_name) );
            $this->dashboard_posts_table( $post_type, 'draft', $title, 'ASC', 3, $drafts );

            echo '</div>';
        }

        function dashboard_posts_status( $post_type ) {
            $counts = get_object_vars( wp_count_posts( $post_type->name ) );
            $total = 0;

            $output = [];
            $post_statuses = get_post_stati( [ 'internal' => false ], 'objects' );
            foreach ( $post_statuses as $post_status ) {
                if ( !empty( $counts[$post_status->name] ) ) {
                    $count = $counts[$post_status->name];
                    $label = sprintf( $post_status->label_count[ $count > 1 ], $count );

                    $params = [ 'post_type' => $post_type->name, 'post_status' => $post_status->name ];
                    $url = $this->build_admin_url( 'edit.php', $params );

                    $output[] = sprintf( '<a href="%s">%s</a>', esc_url( $url ), $label );
                    $total += $count;
                }
            }

            if ( !empty( $output ) ) {
                printf( '<div>' );

                $url = $this->build_admin_url( 'edit.php', [ 'post_type' => $post_type->name ] );
                printf( '<a href="%s">%s (%d)</a>', esc_url( $url ), $post_type->labels->all_items, $total );
                printf( ' · %s · ', join( ' · ', $output ) );

                $url = $this->build_admin_url( 'post-new.php', [ 'post_type' => $post_type->name ] );
                printf( '<a href="%s"><span class="dashicons-before dashicons-plus">%s</span></a>',
                        esc_url( $url ), __( 'New', 'rs-dashboard-widgets' ) );

                printf( '</div>' );

            }

        }

        function dashboard_posts_table( $post_type, $post_status, $title, $order = 'DESC', $limit = 3, $exclude = [] ) {
            $posts = get_posts( [
                'post_type' => $post_type->name,
                'post_status' => $post_status,
                'orderby' => 'post_date',
                'order' => $order,
                'numberposts' => $limit,
                'post__not_in' => $exclude,
            ] );


            $format =  get_option('date_format') . ', ' . get_option('time_format');

            if ( !empty( $posts ) ) {
                printf( '<p><strong>%s</strong></p>', esc_html( $title ) );

                echo '<table>';
                foreach ( $posts as $post )
                    printf( '<tr><td>%s</td><td><a href="%s">%s</a></td><td>%d words</td<</tr>',
                            $this->relative_date( get_post_datetime( $post ) ),
                            $this->build_admin_url( 'post.php', [ 'post' => $post->ID, 'action' => 'edit' ] ),
                            esc_html( $post->post_title ),
                            str_word_count( strip_tags( $post->post_content ) )
                    );

                echo '</table>';
            }

            return wp_list_pluck( $posts, 'ID' );
        }

        // CSS for the dashboard widget
        function do_admin_head_action_dashboard_css() {
            $screen = get_current_screen();

            if ( isset( $screen ) && 'dashboard' == $screen->base ) {
                echo '<style type="text/css">';
                echo ".post-type-status-widget table { width: 100%; }";
                echo ".post-type-status-widget table tr:nth-child(odd) { background: #f6f7f7; }";
                echo ".post-type-status-widget table tr td:nth-child(1) { width: 20%; }";
                echo ".post-type-status-widget table tr td:nth-child(3) { width: 15%; text-align: right; }";
                echo '</style>';
            }
        }

        protected function relative_date( $datetime ) {
            $now = new \DateTime( );
            $dt = $now->diff( $datetime );

            if ( $dt->y ) {
                if ( $dt->m >= 6) $dt->y++;

                if (  $dt->invert )
                    $format = _n( '%d year ago', '%d years ago', $dt->y, 'rs-dashboard-widgets' );
                else
                    $format = _n( 'in %d year', 'in %d years', $dt->y, 'rs-dashboard-widgets' );

                $relative_date = sprintf( $format, $dt->y );
            } elseif ( $dt->m ) {
                if ( $dt->d >= 15) $dt->m++;

                if (  $dt->invert )
                    $format = _n( '%d month ago', '%d months ago', $dt->m, 'rs-dashboard-widgets' );
                else
                    $format = _n( 'in %d month', 'in %d months', $dt->m, 'rs-dashboard-widgets' );

                $relative_date = sprintf( $format, $dt->m );
            } elseif ( $dt->d >= 13 ) {
                if ( $dt->h >= 12) $dt->d++;

                $weeks = (int)( ( $dt->d + 3 ) / 7 );
                if (  $dt->invert )
                    $format = _n( '%d week ago', '%d weeks ago', $weeks, 'rs-dashboard-widgets' );
                else
                    $format = _n( 'in %d week', 'in %d weeks', $weeks, 'rs-dashboard-widgets' );

                $relative_date = sprintf( $format, $weeks );
            } elseif ( $dt->d ) {
                if ( $dt->h >= 12) $dt->d++;

                if (  $dt->invert )
                    $format = _n( '%d day ago', '%d days ago', $dt->d, 'rs-dashboard-widgets' );
                else
                    $format = _n( 'in %d day', 'in %d days', $dt->d, 'rs-dashboard-widgets' );

                $relative_date = sprintf( $format, $dt->d );
            } elseif ( $dt->h ) {
                if ( $dt->m >= 30) $dt->h++;

                if (  $dt->invert )
                    $format = _n( '%d hour ago', '%d hours ago', $dt->h, 'rs-dashboard-widgets' );
                else
                    $format = _n( 'in %d hour', 'in %d hours', $dt->h, 'rs-dashboard-widgets' );

                $relative_date = sprintf( $format, $dt->h );
            } elseif ( $dt->i ) {
                if ( $dt->s >= 30) $dt->i++;

                if (  $dt->invert )
                    $format = _n( '%d minute ago', '%d minutes ago', $dt->i, 'rs-dashboard-widgets' );
                else
                    $format = _n( 'in %d minute', 'in %d minutes', $dt->i, 'rs-dashboard-widgets' );

                $relative_date = sprintf( $format, $dt->i );
            } else
                  $relative_date = __( 'now', 'rs-dashboard-widgets' );

            return $relative_date;
        }

        /************************************************************************
         *
         *	Main RSS/Atom site feed
         *
         ************************************************************************/

        function do_pre_get_posts_action_rss_feed( $wp_query ) {
            if ( $wp_query->is_main_query() && ! is_admin() ) {
                $queried_object = $wp_query->get_queried_object();
                if ( !$queried_object && is_feed() ) {
                    $post_types = $this->get_post_types_for_rss_feed();
                    $wp_query->set( 'post_type', array_map( fn($pto) => $pto->name, $post_types ) );
                }
            }
        }


        /*************************************************************
         *
         * At a glance - Site Health
         *
         *************************************************************/

        function do_dashboard_glance_items_filter_site_health( $items ) {
            $issues = get_transient( 'health-check-site-status-result' );
            if ( $issues === false ) {
                $items[] = sprintf( '<a class="at-a-glance-site-health-warning" href="%s">%s</a>',
                                    esc_url( admin_url( 'site-health.php') ),
                                    esc_html( __( 'Check site health!', 'rs-dashboard-widgets' ) )
                );
            } else {
                $counts = json_decode( $issues, true );
                if ( empty( $counts ) || !is_array( $counts ) )
                    return $items;

                $count = 0;
                if ( array_key_exists( 'critical', $counts ) && intval( $counts['critical'] ) > 0 ) {
                    $count = intval( $counts['critical'] );
                    $label = _n( 'Critical site health issue', 'Critical site health issues', $count, 'rs-dashboard-widgets' );
                    $class = 'at-a-glance-site-health-critical';
                }
                else if ( array_key_exists( 'recommended', $counts ) && intval( $counts['recommended'] ) > 0 ) {
                    $count = intval( $counts['recommended'] );
                    $label = _n( 'Site health warning', 'Site health warnings', $count, 'rs-dashboard-widgets' );
                    $class = 'at-a-glance-site-health-warning';
                }

                if ( $count ) {
                    $items[] = sprintf( '<a class="%s" href="%s">%d %s</a>',
                                        esc_attr( $class ),
                                        esc_url( admin_url( 'site-health.php') ),
                                        $count, esc_html( $label )
                    );

                }
            }
            return $items;
        }

        // Add the correct icons to the At a Glance dashboard widget
        function do_admin_head_action_site_health() {
            echo '<style type="text/css">';
            printf( '#dashboard_right_now li a.at-a-glance-site-health-critical::before { content: "\%x"; }', DashIcons::lookup( 'flag' ) );
            printf( '#dashboard_right_now li a.at-a-glance-site-health-warning::before { content: "\%x"; }', DashIcons::lookup( 'warning' ) );
            echo '</style>';
        }


        /************************************************************************
         *
         *	Add site title to admin headers
         *
         ************************************************************************/

        // Make the simple columns narrower
        function do_admin_head_action_site_title() {
            if ( $this->get_option( 'blogname_in_titles' ) ) {
                echo '<style type="text/css">';
                printf( '.wrap H1::after { content: " — %s" }', get_option( 'blogname' ) );
                echo '</style>';
            }
        }

        /************************************************************************
         *
         *	Settings
         *
         ************************************************************************/

        function do_admin_menu_action() {
            $this->settings_add_menu(
                __( 'RS Dashboard Widgets', 'rs-dashboard-widgets' ),
                __( 'RS Dashboard', 'rs-dashboard-widgets' )
            );
        }

        protected $rs_dashboard_widgets_blacklist_callback_priority = 99;
        function do_rs_dashboard_widgets_blacklist_filter( $post_types ){
            return array_filter( $post_types, fn( $pt ) => ! in_array( $pt, $this->post_type_blacklist ) );
        }

        protected $rs_dashboard_widgets_post_type_viewable_callback_priority = 99;
        function do_rs_dashboard_widgets_post_type_viewable_filter( $post_types ){
            return array_filter( $post_types, fn( $pt ) => is_post_type_viewable( $pt ) );
        }

        function settings_define_sections_and_fields() {

            // Blogname in section titles

            $section = $this->settings_add_section(
                'section_general_options',
                __( 'General RS Dashboard options', 'rs-dashboard-widgets' ),
                __( 'Overall control of RS Dashboard behaviour.', 'rs-dashboard-widgets' )
            );

            $this->settings_add_field(
                'blogname_in_titles', $section,
                __( 'Add site name to admin titles', 'rs-dashboard-widgets' ),
                'settings_field_checkbox_html'
            );


            // Post type choices for addition to standard widgets

            $post_types = $this->settings_prepare_post_types_menu(
                [ 'public' => true, '_builtin' => false ],
                [
                    'rs_dashboard_widgets_at_a_glance',
                    'rs_dashboard_widgets_everywhere',
                    'rs_dashboard_widgets_blacklist',
                ]
            );

            if ( $post_types  ) {
                $section = $this->settings_add_section(
                    'section_std_widgets',
                    __( 'Standard widgets', 'rs-dashboard-widgets' ),
                    __( 'Select which post types to add to the At-a-Glance and the Activity widgets', 'rs-dashboard-widgets' )
                );

                $this->settings_add_post_types_menu(
                    'post_types_std_widgets', $section,
                    __( 'Post-types', 'rs-dashboard-widgets' ),
                    $post_types
                );
            }


            // Post types choices for new widgets

            $section = $this->settings_add_section(
                'section_new_widgets',
                __( 'New post type status widgets', 'rs-dashboard-widgets' ),
                __( 'Select for which post types to add new status widgets', 'rs-dashboard-widgets' )
            );

            $this->settings_build_post_types_menu(
                'post_types_new_widgets', $section,
                __( 'Post-types', 'rs-dashboard-widgets' ),
                [ 'public' => true ], [],
                [
                    'rs_dashboard_widgets_status_widget',
                    'rs_dashboard_widgets_everywhere',
                    'rs_dashboard_widgets_blacklist',
                ]
            );

            // Post types for unified search

            $section = $this->settings_add_section(
                'section_unified_search',
                __( 'Unified Search', 'rs-dashboard-widgets' ),
                __( 'Select for which post types to included in the unified search', 'rs-dashboard-widgets' )
            );

            $this->settings_build_post_types_menu(
                'post_types_search', $section,
                __( 'Post-types', 'rs-dashboard-widgets' ),
                [ 'public' => true ], [],
                [
                    'rs_dashboard_widgets_unified_search',
                    'rs_dashboard_widgets_everywhere',
                    'rs_dashboard_widgets_blacklist',
                ]
            );

            // Post types for extended RSS feed

            $section = $this->settings_add_section(
                'section_rss_feed',
                __( 'Post types to include in main RSS/Atom feed', 'rs-dashboard-widgets' ),
                __( 'Select for which post types to include in the main RSS/Atom feed', 'rs-dashboard-widgets' )
            );

            $this->settings_build_post_types_menu(
                'post_types_rss_feed', $section,
                __( 'Post-types', 'rs-dashboard-widgets' ),
                [ 'public' => true ], [],
                [
                    'rs_dashboard_widgets_rss_feed',
                    'rs_dashboard_widgets_everywhere',
                    'rs_dashboard_widgets_post_type_viewable',
                    'rs_dashboard_widgets_blacklist',
                ]
            );


            // Taxonomies for quick links

            $section = $this->settings_add_section(
                'section_taxonomies_widget',
                __( 'Taxonomies for quick-links', 'rs-dashboard-widgets' ),
                __( 'Select for which taxonomies to include in quick links widget', 'rs-dashboard-widgets' )
            );

            $this->settings_build_taxonomies_menu(
                'taxonomies_quick_links', $section,
                __( 'Taxonomies', 'rs-dashboard-widgets' ),
                [ 'public' => true ]
            );
        }


        /************************************************************************
         *
         *	Admin bar sub-menu for creating terms
         *
         ************************************************************************/

        protected $admin_bar_menu_new_terms_menu_callback_priority = 71;

        function do_admin_bar_menu_action_new_terms_menu( $wp_admin_bar ) {

            $wp_admin_bar->add_node( [
        		'id'     => 'new-post_tag',
        		'title'  => '<span class="ab-icon dashicons-plus" aria-hidden="true" style="top: 4px;"></span><span class="ab-label">Tag</span>',
        		'href'   => esc_url( admin_url( 'edit-tags.php?taxonomy=post_tag' ) ),
        		'meta'   => false
            ] );

            $taxonomies = $this->get_option( 'taxonomies_quick_links', [] );
            if ( empty( $taxonomies ) )
                return;


            foreach ( $taxonomies as $taxonomy ) {
                $tax = get_taxonomy( $taxonomy );
                if ( empty( $tax ) ) continue;

                $labels = get_taxonomy_labels( $tax );
                if ( empty( $labels ) ) continue;

                $wp_admin_bar->add_node( [
                    'parent' => 'new-post_tag',
                    'id'     => "new-$taxonomy",
                    'title'  => $labels->singular_name,
                    'href'   => esc_url( admin_url( 'edit-tags.php?taxonomy=' . $taxonomy ) ),
                    'meta'   => false
                ] );
            }
        }

        /************************************************************************
         *
         *	Dashboard search box across post types
         *
         ************************************************************************/

        function do_wp_dashboard_setup_action_search() {
            wp_add_dashboard_widget(
                "unified-search-dashboard-widget",
                esc_html__( 'Unified search', 'rs-dashboard-widgets' ),
                [ $this, 'unified_search_render' ]
            );
        }

        function unified_search_render() {
            $url =  admin_url( 'admin.php' );
            printf( '<form method="get" action="%s">', esc_url( $url ) );
            printf( '<input type="hidden" name="page" value="%s" />', esc_attr( $this->search_page_name ) );
            printf( '<div>%s</div>', esc_html__( 'Enter search terms:', 'rs-dashboard-widgets' ) );
            printf( '<input type="text" name="s" value="" />' );
            printf( '<input type="submit" id="unified-search-submit" class="button" value="Search">' );
            printf( '</form>' );

            $post_types = $this->get_post_types_for_unified_search();

            $links = [];
            $stati = get_post_stati( [ 'exclude_from_search' => false, 'internal' => false, ] );
            foreach ( $stati as $status ) {
                $ids = get_posts( [
                    'post_type' => array_map( fn($pto) => $pto->name, $post_types ),
                    'post_status' => $status,
                    'fields' => 'ids',
                    'numberposts' => -1,
                ] );

                if ( count( $ids ) ) {
                    $url = $this->build_admin_url( 'admin.php', [
                        'page' => $this->search_page_name,
                        'post_status' => $status,
                    ] );

                    $status_object = get_post_status_object( $status );
                    $label = sprintf( _n( $status_object->label_count['singular'], $status_object->label_count['plural'], count( $ids ), 'rs-dashboard-widgets' ), count( $ids ) );

                    $links[ $status ] = sprintf( '<a href="%s">%s</a>', esc_url( $url ), $label );
                }
            }

            if ( $links )
                printf( '<p>%s: %s.</p>', __( 'By status', 'rs-dashboard-widgets' ), join( ' · ', $links ) );

            $links = array_map(
                fn($pto) => sprintf( '<a href="%s?post_type=%s">%s</a>', admin_url( 'edit.php' ), $pto->name, $pto->labels->name ),
                $post_types
            );

            if ( $links )
                printf( '<p>Post types: %s</p>', join( ', ', $links ) );

            $links = [];

            $taxonomies = $this->get_option( 'taxonomies_quick_links', [] );
            if ( ! empty( $taxonomies ) ) {
                foreach ( $taxonomies as $taxonomy ) {
                    $tax = get_taxonomy( $taxonomy );
                    if ( empty( $tax ) ) continue;

                    $labels = get_taxonomy_labels( $tax );
                    if ( ! empty( $labels ) )
                        $links[] = sprintf( '<a href="%s?taxonomy=%s">%s</a>', admin_url( 'edit-tags.php' ), $taxonomy, $labels->name );
                }
            }

            if ( $links )
                printf( '<p>Taxonomies: %s</p>', join( ', ', $links ) );
        }



        /************************************************************************
         *
         *	Lists of search result - using Dashboard_Search_Post_Table
         *
         ************************************************************************/

        protected $search_page = NULL;
        protected $search_page_name = 'unified_search';

        function do_admin_menu_action_search_page() {
            $this->search_page = add_menu_page(
                __('Search', 'rs-dashboard-widgets' ),
                __('Search', 'rs-dashboard-widgets' ),
                'edit_posts',
                $this->search_page_name,
                [ $this, 'search_page_render' ],
                'dashicons-search',
                3
            );
            // Position: 3 under Dashboard; 59 under comments or CPTs;

            add_action( "load-{$this->search_page}", [ $this, 'load_search_page_hook' ] );
        }

        function search_page_render() {
            if ( ! current_user_can( 'edit_posts' ) )
                wp_die( 'Not allowed' );

            printf( '<div class="wrap"><h1>%s</h1>', __( 'Unified Search', 'rs-dashboard-widgets' ) );

            // with POST pagination and search doesn't work
            echo '<form method="get">';
            printf( '<input type="hidden" name="page" value="%s" />', esc_attr( $this->search_page_name ) );

            $table = new Dashboard_Search_Post_Table( $this->search_page, $this->get_post_types_for_unified_search() );
            $table->prepare_items();
            $table->output_search_term();
            $table->views();
            $table->search_box( __( 'Search', 'rs-dashboard-widgets' ), $this->search_page_name );
            $table->display();

            $js = 'jQuery(document).ready(function($) { $( ".search-box #unified_search-search-input" ).each( function () { $(this).focus(); } ); } );';
            echo "<script type='text/javascript'>{$js}</script>";

            echo '</form>';
            echo '</div>';
        }

        // Set up the Screen Options

        function load_search_page_hook() {
            $screen = get_current_screen();

            // get out of here if we are not on our settings page
            if( !is_object( $screen ) || $screen->id != $this->search_page )
                return;

            $args = array(
                'label' => __( 'Elements per page', 'rs-dashboard-widgets' ),
                'default' => 20,
                'option' => 'elements_per_page'
            );

            add_screen_option( 'per_page', $args );

            // This allows column selection - I have no idea why
            new Dashboard_Search_Post_Table( $this->search_page, [] );
        }

        protected $set_screen_option_callback_hook = 'set-screen-option';

        function do_set_screen_option_filter( $status, $option, $value ) {
            return $value;
        }

        // Make the simple columns narrower
        function do_admin_head_action_link_table_css() {
            echo '<style type="text/css">';
            echo ".{this->search_page} .column-status { width: 10% !important; overflow: hidden }";
            echo ".{this->search_page} .column-type { width: 10% !important; overflow: hidden }";
            echo ".{this->search_page} .column-date { width: 14% !important; overflow: hidden; }";
            echo '</style>';
        }


        /************************************************************************
         *
         *	Issues Taxonomy widget rows
         *
         ************************************************************************/

        function do_issues_taxonomy_row_link_filter( $link, $term ) {
            return $this->build_admin_url( 'admin.php', [
                'page' => $this->search_page_name,
                'issue' => $term->slug,
                'post_status' => 'any',
            ] );
        }
    }

    DashboardWidgets::instance();
} );
