<?php

namespace ReneSeindal;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * RS Link Checker plugin CLI commands
 *
 * ## EXAMPLES
 *
 *     wp link-checker update
 *
 * @when after_wp_load
 */

class LinkCheckerCLI {
    protected $plugin = NULL;

	function __construct() {
        global $RSLinkCheckerPlugin;
        $this->plugin = $RSLinkCheckerPlugin;
    }
}

class LinkCheckerCLI_post extends LinkCheckerCLI {

    /**
     * Scan post for links but don't store them
     *
     * ## OPTIONS
     *
     * <post-id>
     * : Scan post <post-id>
     * ---
     *
     * [--format=<format>]
     * : Render output in a particular format.
     * ---
     * default: table
     * options:
     *   - table
     *   - csv
     *   - json
     *   - count
     *   - yaml
     * ---
     *
     *
     * ## EXAMPLES
     *
     *     wp link-checker post test-scan 42
     *
     * @subcommand test-scan
     */

    public function test_scan( $args, $assoc ) {

        $id = intval( $args[0] );
		if ( $id <= 0 ) {
			\WP_CLI::error( 'Argument <post-id> is missing' );
			return;
		}

        $post = get_post( $id );
		if ( ! $post ) {
			\WP_CLI::error( 'Post <post-id> not found' );
			return;
		}

        $links = $this->plugin->get_post_aggregate_links( $post );
		$output = [];

		foreach ( $links as $link => $count ) {
			$output[] = [
				__('URL', 'rs-link-checker') => $link,
				__('Count', 'rs-link-checker') => $count,
			];
		}

		if ( $output ) {
			\WP_CLI\Utils\format_items( $assoc['format'], $output, join(',', array_keys($output[0])) );
		}
    }

    /**
     * Scan and store all links in a post
     *
     * ## OPTIONS
     *
     * <post-id>
     * : Scan post <post-id>
     * ---
     *
     * [--format=<format>]
     * : Render output in a particular format.
     * ---
     * default: table
     * options:
     *   - table
     *   - csv
     *   - json
     *   - count
     *   - yaml
     * ---
     *
     *
     * ## EXAMPLES
     *
     *     wp link-checker post scan 42
     *
     */

    public function scan( $args, $assoc ) {

        $id = intval( $args[0] );
		if ( $id <= 0 ) {
			\WP_CLI::error( 'Argument <post-id> is missing' );
			return;
		}

        $post = get_post( $id );
		if ( ! $post ) {
			\WP_CLI::error( 'Post <post-id> not found' );
			return;
		}

        $links = $this->plugin->get_post_links( $post );
		$output = [];

		foreach ( $links as $link ) {
			$output[] = [
				__('URL', 'rs-link-checker') => $link,
			];
		}

		if ( $output ) {
			\WP_CLI\Utils\format_items( $assoc['format'], $output, join(',', array_keys($output[0])) );
		}

        $this->plugin->save_post_links( $post, $links, true );
        \WP_CLI::success( 'Done' );
    }


    /**
     * Show rescan queue of posts
     *
     * ## OPTIONS
     *
     * [--format=<format>]
     * : Render output in a particular format.
     * ---
     * default: table
     * options:
     *   - table
     *   - csv
     *   - json
     *   - count
     *   - yaml
     * ---
     *
     *
     * ## EXAMPLES
     *
     *     wp link-checker post queue
     *
     */

    public function queue( $args, $assoc ) {

        $posts = $this->plugin->get_post_queue( -1 );

        if ( empty( $posts ) ) {
            \WP_CLI::warning( 'No posts found' );
            return;
        }

        $output = [];
        foreach ( $posts as $post ) {

            $last_scan = $post->_rslc_last_check;
            if ( $last_scan )
                $last = wp_date( 'Y-m-d H:i:s', $last_scan );
            else
                $last = __( 'Never', 'rs-link-checker' );

            $output[] = [
                __('ID', 'rs-link-checker') => $post->ID,
                __('Type', 'rs-link-checker') => $post->post_type,
                __('Status', 'rs-link-checker') => $post->post_status,
                __('Last scan', 'rs-link-checker') => $last,
                __('Title', 'rs-link-checker') => $post->post_title,
            ];
        }

        if ( $output ) {
            \WP_CLI\Utils\format_items( $assoc['format'], $output, join(',', array_keys($output[0])) );
        }
    }

    /**
     * Rescan all posts
     *
     * ## OPTIONS
     *
     * [--batch=<count>]
     * : Use a batch size of <count>
     * ---
     * default: from plugin settings
     * ---
     *
     * [--format=<format>]
     * : Render output in a particular format.
     * ---
     * default: table
     * options:
     *   - table
     *   - csv
     *   - json
     *   - count
     *   - yaml
     * ---
     *
     *
     * ## EXAMPLES
     *
     *     wp link-checker post rescan
     *
     */

    public function rescan( $args, $assoc ) {
        $batch_size = intval( $assoc['batch'] );

        $posts = $this->plugin->get_post_queue( $batch_size );

        if ( empty( $posts ) ) {
            \WP_CLI::warning( 'No posts found' );
            return;
        }

        foreach ( $posts as $post ) {
            \WP_CLI::line( 'Parsing post #' . $post->ID . ': ' . $post->post_title );

            $output = [];

            $links = $this->plugin->get_post_links( $post );

            if ( ! empty( $links ) ) {
                foreach ( $links as $link ) {
                    $output[] = [
                        __('URL', 'rs-link-checker') => $link,
                    ];
                }

                if ( $output ) {
                    \WP_CLI\Utils\format_items( $assoc['format'], $output, join(',', array_keys($output[0])) );
                }

            }
            $this->plugin->save_post_links( $post, $links );
        }
    }


}

\WP_CLI::add_command( 'link-checker post', new LinkCheckerCLI_post() );


/************************************************************************/


class LinkCheckerCLI_link extends LinkCheckerCLI {

    /**
     * List all registered links
     *
     * ## OPTIONS
     *
     * [--status=<status>]
     * : Show only links of a particular status
     * ---
     * default: broken
     * options:
     *   - all
     *   - unchecked
     *   - ok
     *   - redirect
     *   - broken
     *   - dismissed
     *   - blacklisted
     *   - ignored
     * ---
     *
     * [--format=<format>]
     * : Render output in a particular format.
     * ---
     * default: table
     * options:
     *   - table
     *   - csv
     *   - json
     *   - count
     *   - yaml
     * ---
     *
     *
     * ## EXAMPLES
     *
     *     wp link-checker link list --status=broken
     *
     */

    public function list( $args, $assoc ) {
        $terms = $this->plugin->get_links_filtered( $assoc['status'] ?: 'broken' );
        $this->list_links( $terms, $assoc['format'] );
    }

    private function list_links( $terms, $format ) {
		$output = [];

		foreach ( $terms as $term ) {
            $term_id = $term->term_id;

            $meta = $this->plugin->get_term_meta( $term );

            $date = '';
            if ( isset( $meta['next_check'] ) )
                $date = wp_date( 'Y-m-d H:i:s', $meta['next_check'] );

			$output[] = [
				__('ID', 'rs-link-checker') => $term_id,
				__('Next', 'really-simplet-link-checker') => $date,
				__('Status', 'rs-link-checker') => $meta['status'] ?? '-',
				__('Message', 'rs-link-checker') => $meta['message'] ?? '-',
				__('Used', 'rs-link-checker') => $term->count,
				__('URL', 'rs-link-checker') => $meta['url'],
				__('Redirect', 'rs-link-checker') => $meta['redirect'] ?? '-',
				__('Dismissed', 'rs-link-checker') => $meta['dismissed'] ?? '-',
			];
		}

		if ( $output ) {
			\WP_CLI\Utils\format_items( $format, $output, join(',', array_keys($output[0])) );
		}
    }

    /**
     * Show link check queue
     *
     * ## OPTIONS
     *
     * [--format=<format>]
     * : Render output in a particular format.
     * ---
     * default: table
     * options:
     *   - table
     *   - csv
     *   - json
     *   - count
     *   - yaml
     * ---
     *
     *
     * ## EXAMPLES
     *
     *     wp link-checker link queue
     *
     */

    public function queue( $args, $assoc ) {
        $terms = $this->plugin->get_link_queue( );
        $this->list_links( $terms, $assoc['format'] );
    }


    /**
     * Check a single link by term id or MD5 slug
     *
     * ## OPTIONS
     *
     * <slug-or-term_id>...
     * : Term id or slug MD5 hash
     * ---
     *
     * [--format=<format>]
     * : Render output in a particular format.
     * ---
     * default: table
     * options:
     *   - table
     *   - csv
     *   - json
     *   - count
     *   - yaml
     * ---
     *
     *
     * ## EXAMPLES
     *
     *     wp link-checker link check fd20658c29be8867f851bdcf44b815f7 789
     *
     */

    public function check( $args, $assoc ) {

        $done = [];

        foreach ( $args as $slug ) {
            if ( 32 == strlen( $slug ) )
                $term = get_term_by( 'slug', $slug, $this->plugin->get_taxonomy_name() );
            else
                $term = get_term( intval( $slug ), $this->plugin->get_taxonomy_name() );

            if ( ! $term ) {
                \WP_CLI::error( 'Term not found' );
                return;
            }

            if ( array_key_exists( $term->term_id, $done ) )
                continue;


            \WP_CLI::line( 'Checking link #' . $term->term_id );
            $this->plugin->check_link( $term );

            $updated = get_term( $term->term_id, $this->plugin->get_taxonomy_name() );

            if ( $updated )
                $done[ $term->term_id ] = $updated;
        }

        $this->list_links( $done, $assoc['format'] );
    }

    /**
     * Chect a batch of links
     *
     * ## OPTIONS
     *
     * [--format=<format>]
     * : Render output in a particular format.
     * ---
     * default: table
     * options:
     *   - table
     *   - csv
     *   - json
     *   - count
     *   - yaml
     * ---
     *
     *
     * ## EXAMPLES
     *
     *     wp link-checker link batch
     *
     */

    public function batch( $args, $assoc ) {
        $terms = [];

        $callback = function( $term_id ) use ( &$terms ) {
            $term = get_term( $term_id, $this->plugin->get_taxonomy_name() );
            if ( $term ) {
                $meta = $this->plugin->get_term_meta( $term );
                \WP_CLI::log( sprintf( "Term %s Status %d %s URL %s", $term_id, $meta['status'], $meta['message'], $meta['url'] ) );
                $terms[] = $term;
            } else {
                \WP_CLI::log( sprintf( "Term %s deleted", $term_id ) );
            }
        };

        $this->plugin->link_check_batch( $callback );

        $this->list_links( $terms, $assoc['format'] );
    }

    /**
     * Show everything about a single link
     *
     * ## OPTIONS
     *
     * <slug-or-term_id>
     * : Term id or slug MD5 hash
     * ---
     *
     * [--format=<format>]
     * : Render output in a particular format.
     * ---
     * default: table
     * options:
     *   - table
     *   - csv
     *   - json
     *   - count
     *   - yaml
     * ---
     *
     *
     * ## EXAMPLES
     *
     *     wp link-checker link show fd20658c29be8867f851bdcf44b815f7
     *     wp link-checker link show 748
     *
     */

    public function show( $args, $assoc ) {
        $slug = $args[0];

        if ( 32 == strlen( $slug ) )
            $term = get_term_by( 'slug', $slug, $this->plugin->get_taxonomy_name() );
        else
            $term = get_term( intval( $slug ), $this->plugin->get_taxonomy_name() );

        if ( ! $term ) {
            \WP_CLI::error( 'Term not found' );
            return;
        }

        $meta = $this->plugin->get_term_meta( $term );

        $date = '';
        if ( isset( $meta['next_check'] ) )
            $date = wp_date( 'Y-m-d H:i:s', $meta['next_check'] );

        $output = [
            [
            __('ID', 'rs-link-checker') => $term->term_id,
            __('Slug', 'rs-link-checker') => $term->slug,
            __('Next', 'really-simplet-link-checker') => $date,
            __('Status', 'rs-link-checker') => $meta['status'] ?? '-',
            __('Message', 'rs-link-checker') => $meta['message'] ?? '-',
            __('Used', 'rs-link-checker') => $term->count,
            __('URL', 'rs-link-checker') => $meta['url'],
            __('Redirect', 'rs-link-checker') => $meta['redirect'] ?? '',
            __('Dismissed', 'rs-link-checker') => $meta['dismissed'] ?? '',
			]
        ];

        \WP_CLI\Utils\format_items( $assoc['format'], $output, join(',', array_keys($output[0])) );
    }

    /**
     * List posts using a link
     *
     * ## OPTIONS
     *
     * <slug-or-term_id>
     * : Term id or slug MD5 hash
     * ---
     *
     * [--format=<format>]
     * : Render output in a particular format.
     * ---
     * default: table
     * options:
     *   - table
     *   - csv
     *   - json
     *   - count
     *   - yaml
     * ---
     *
     *
     * ## EXAMPLES
     *
     *     wp link-checker link posts fd20658c29be8867f851bdcf44b815f7
     *     wp link-checker link posts 748
     *
     */

    public function posts( $args, $assoc ) {
        $slug = $args[0];

        if ( 32 == strlen( $slug ) )
            $term = get_term_by( 'slug', $slug, $this->plugin->get_taxonomy_name() );
        else
            $term = get_term( intval( $slug ), $this->plugin->get_taxonomy_name() );

        if ( ! $term ) {
            \WP_CLI::error( 'Term not found' );
            return;
        }

        $posts = $this->plugin->get_link_posts( $term );

        $output = [];
        foreach ( $posts as $post ) {
            $output[] = [
                __('ID', 'rs-link-checker') => $post->ID,
                __('Type', 'rs-link-checker') => $post->post_type,
                __('Status', 'rs-link-checker') => $post->post_status,
                __('Title', 'rs-link-checker') => $post->post_title,
            ];
        }

        if ( $output ) {
            \WP_CLI\Utils\format_items( $assoc['format'], $output, join(',', array_keys($output[0])) );
        }
    }


    /**
     * Update a redirecting link in all posts
     *
     * ## OPTIONS
     *
     * <slug-or-term_id>
     * : Term id or slug MD5 hash
     * ---
     *
     * [--new-url=<url>]
     * : Uuse <url> for new link destination
     * ---
     *
     * ## EXAMPLES
     *
     *     wp link-checker link update fd20658c29be8867f851bdcf44b815f7
     *     wp link-checker link update 748
     *
     */

    public function update( $args, $assoc ) {
        $slug = $args[0];

        if ( 32 == strlen( $slug ) )
            $term = get_term_by( 'slug', $slug, $this->plugin->get_taxonomy_name() );
        else
            $term = get_term( intval( $slug ), $this->plugin->get_taxonomy_name() );

        if ( ! $term ) {
            \WP_CLI::error( 'Term not found' );
            return;
        }

        $this->plugin->update_link_posts( $term, $assoc['new-url'] ?? NULL );
    }


    /**
     * Rescan all posts for a link
     *
     * ## OPTIONS
     *
     * <slug-or-term_id>
     * : Term id or slug MD5 hash
     * ---
     *
     * ## EXAMPLES
     *
     *     wp link-checker link rescan fd20658c29be8867f851bdcf44b815f7
     *     wp link-checker link rescan 748
     *
     */

    public function rescan( $args, $assoc ) {
        $slug = $args[0];

        if ( 32 == strlen( $slug ) )
            $term = get_term_by( 'slug', $slug, $this->plugin->get_taxonomy_name() );
        else
            $term = get_term( intval( $slug ), $this->plugin->get_taxonomy_name() );

        if ( ! $term ) {
            \WP_CLI::error( 'Term not found' );
            return;
        }

        $this->plugin->rescan_link_posts( $term );
    }





    /**
     * Purge stale links
     *
     * ## EXAMPLES
     *
     *     wp link-checker link purge
     *
     */

    public function purge( $args, $assoc ) {
        $terms = $this->plugin->get_links_filtered( 'stale' );
        $terms = array_filter( $terms, fn( $term ) => $term->count == 0 );


        $this->list_links( $terms, 'table' );

        $this->plugin->purge_stale_links( $terms );
    }

    /**
     * Check blacklist
     *
     * ## EXAMPLES
     *
     *     wp link-checker link check-blacklist
     *
     * @subcommand check-blacklist
     */

    public function check_blacklist( $args, $assoc ) {
        $terms = $this->plugin->get_link_list( );

        foreach ( $terms as $term ) {
            $url = $this->plugin->get_term_meta( $term, 'url' );
            if ( $url && $this->plugin->url_is_blacklisted( $url ) ) {
                if ( $this->plugin->link_is_blacklisted( $term ) ) {
                    \WP_CLI::line( "Already blacklisted term {$term->term_id}: {$url}");
                } else {
                    \WP_CLI::line( "Blacklisting term {$term->term_id}: {$url}");
                    $this->plugin->update_term_meta( $term, 'dismissed', 'blacklisted' );
                    $this->plugin->update_term_meta( $term, 'next_check', false );
                }
            } elseif ( $this->plugin->link_is_blacklisted( $term ) ) {
                \WP_CLI::line( "Un-blacklisting term {$term->term_id}: {$url}");
                $this->plugin->update_term_meta( $term, 'dismissed', false );
            }
        }

    }


    /**
     * Send mail with broken links
     *
     * ## EXAMPLES
     *
     *     wp link-checker link mail
     *
     */

    public function mail( $args, $assoc ) {
        $terms = $this->plugin->send_mail_with_broken_links( );
    }



}

\WP_CLI::add_command( 'link-checker link', new LinkCheckerCLI_link() );



/************************************************************************/



class LinkCheckerCLI_top extends LinkCheckerCLI {

    /**
     * Show plugin status
     *
     * ## EXAMPLES
     *
     *     wp link-checker status
     *
     */


    public function status( $args, $assoc ) {
        \WP_CLI::line( $this->plugin->post_scan_status() );
        \WP_CLI::line( $this->plugin->scheduler_status() );

        $counts = $this->plugin->link_group_counts();
        $labels = $this->plugin->link_group_labels();

        $output = [];
        foreach ( $labels as $key => $label ) {
            if ( 0 == $counts[ $key ] && !( $assoc['all'] ?? false ) )
                continue;

            $output[] = [
				__('Status', 'rs-link-checker') => $label,
				__('Count', 'rs-link-checker') => $counts[$key],
			];
		}

		if ( $output ) {
			\WP_CLI\Utils\format_items( 'table', $output, join(',', array_keys($output[0])) );
		}
    }

    /**
     * Show plugin options
     */

    public function options( $args, $assoc ) {
        print_r( $this->plugin->get_options() );
    }

    /**
     * Reset everything
     */

    public function reset( $args, $assoc ) {
        $this->plugin->reset_posts();
        $this->plugin->reset_links();
    }
}

\WP_CLI::add_command( 'link-checker', new LinkCheckerCLI_top() );
