<?php
/**
 * Plugin Name:         RS Link Checker
 * Description:         A simple to use link checker that doesn't nag continuously.
 * Plugin URI:          https://plugins.seindal.dk/plugins/rs-link-checker/
 * Update URI:          https://plugins.seindal.dk
 * Author:              René Seindal
 * Author URI:          https://plugins.seindal.dk/
 * Donate link:         https://mypos.com/@historywalks
 * Requires Plugins:    rs-base-plugin
 * License:             GPL v2 or later
 * License URI:         https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:         rs-link-checker
 * Domain Path:         /languages
 * Requires PHP:        7.4
 * Requires at least:   5.0
 * Version:             0.27.3
 **/

namespace ReneSeindal;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

require_once( ABSPATH . 'wp-admin/includes/taxonomy.php' );
require_once( ABSPATH . 'wp-admin/includes/class-wp-list-table.php' );

add_action( 'plugins_loaded', function() {
    class LinkCheckerPlugin extends PluginBase {
        protected $plugin_file = __FILE__;

        use PluginBaseSettings;

        protected $config_taxonomy_name = 'link_checker';

        // Default settings
        protected $config = [
            'post_types' => [ 'post', 'page' ],
            'post_statuses' => [ 'publish' ],
            'post_batch_size' => 20,
            'link_batch_size' => 20,
            'link_interval' => WEEK_IN_SECONDS,
            'link_blacklist_text' => '',
            'schedule' => 'none',
            'logging' => true,
            'user_agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/119.0.0.0 Safari/537.36',
        ];

        // Helpers for WP-CLI functions

        function get_taxonomy_name() {
            return $this->config_taxonomy_name;
        }

        // Getthe post types for the taxonomy

        function get_post_types() {
            $tax = get_taxonomy( $this->config_taxonomy_name );
            return $tax ? $tax->object_type : [];
        }

        /************************************************************************
         *
         *	Link storage in a taxonomy
         *
         ************************************************************************/

        public function do_init_action_setup_taxonomy() {
            $public = false;
            if ( defined( 'WP_CLI' ) && WP_CLI ) {
                $public = true;
            }

            $labels = [
                'name'                       => __( 'Checked links', 'rs-link-checker' ),
                'singular_name'              => _x( 'Checked link', 'taxonomy general name', 'rs-link-checker' ),
                'search_items'               => __( 'Search Checked links', 'rs-link-checker' ),
                'popular_items'              => __( 'Popular Checked links', 'rs-link-checker' ),
                'all_items'                  => __( 'All Checked links', 'rs-link-checker' ),
                'parent_item'                => __( 'Parent Checked link', 'rs-link-checker' ),
                'parent_item_colon'          => __( 'Parent Checked link:', 'rs-link-checker' ),
                'edit_item'                  => __( 'Edit Checked link', 'rs-link-checker' ),
                'update_item'                => __( 'Update Checked link', 'rs-link-checker' ),
                'view_item'                  => __( 'View Checked link', 'rs-link-checker' ),
                'add_new_item'               => __( 'Add New Checked link', 'rs-link-checker' ),
                'new_item_name'              => __( 'New Checked link', 'rs-link-checker' ),
                'separate_items_with_commas' => __( 'Separate Checked links with commas', 'rs-link-checker' ),
                'add_or_remove_items'        => __( 'Add or remove Checked links', 'rs-link-checker' ),
                'choose_from_most_used'      => __( 'Choose from the most used Checked links', 'rs-link-checker' ),
                'not_found'                  => __( 'No Checked links found.', 'rs-link-checker' ),
                'no_terms'                   => __( 'No Checked links', 'rs-link-checker' ),
                'menu_name'                  => __( 'Checked links', 'rs-link-checker' ),
                'items_list_navigation'      => __( 'Checked links list navigation', 'rs-link-checker' ),
                'items_list'                 => __( 'Checked links list', 'rs-link-checker' ),
                'most_used'                  => _x( 'Most Used', 'backlink count', 'rs-link-checker' ),
                'back_to_items'              => __( '&larr; Back to Checked links', 'rs-link-checker' ),
            ];

            register_taxonomy( $this->config_taxonomy_name, [],
                               [
                                   'hierarchical'          => false,
                                   'public'                => $public,
                                   'show_in_nav_menus'     => false,
                                   'show_ui'               => false,
                                   'show_admin_column'     => false,
                                   'query_var'             => false,
                                   'rewrite'               => false,
                                   'labels'                => $labels,
                                   'show_in_rest'          => false,
                                   'update_count_callback' => '_update_generic_term_count',
                               ] );

        }

        // Connect post types - later to other plugisn time to register them

        function do_wp_loaded_action_connect_taxonomy() {
            $taxonoy_post_types = $this->filter_option_list(
                'post_types',
                get_post_types( [ 'public' => true ], 'names' ),
                $this->config['post_types']
            );

            foreach ( $taxonoy_post_types as $post_type )
                register_taxonomy_for_object_type( $this->config_taxonomy_name, $post_type );

            if ( $this->get_option( 'synced_patterns' ) )
                register_taxonomy_for_object_type( $this->config_taxonomy_name, 'wp_block' );
        }

        // Delete all terms
        function reset_links() {
            $args = [
                'taxonomy' => $this->config_taxonomy_name,
                'fields' => 'ids',
            ];

            $term_ids = get_terms( $args );

            foreach ( $term_ids as $term_id ) {
                wp_delete_term( $term_id, $this->config_taxonomy_name );
            }

        }

        // Purge stale links
        function purge_stale_links( $terms = NULL ) {
            if ( NULL == $terms )
                $terms = $this->get_links_filtered( 'stale' );

            $terms = array_filter( $terms, fn( $term ) => $term->count == 0 );

            foreach ( $terms as $term ) {
                wp_delete_term( $term->term_id, $this->config_taxonomy_name );
            }
        }

        // Map urls to taxonomy

        // Problem: term names max 100 characters
        // Solution: use MD5 hash for name/slug and store the rest in meta fields

        // Meta fields:
        // url = url
        // next_check = unix timestamp of next scheduled link check
        // status = HTTP response code, 999 for network errors
        // message = HTTP message or network error message
        // response = entire last response serialised
        // dismissed = user | blacklisted

        function link_to_term( $link ) {
            $slug = md5( $link );

            $term = get_term_by( 'slug', $slug, $this->config_taxonomy_name );
            if ( false !== $term )
                return $term;

            $new = wp_insert_term( $slug, $this->config_taxonomy_name, [
                'description' => $link,
                'slug' => $slug,
            ] );

            if ( is_wp_error( $new ) )
                return false;

            $term_id = $new['term_id'];
            update_term_meta( $term_id, 'url', $link );

            if ( $this->url_is_blacklisted( $link ) )
                $this->dismiss_link( $term, 'blacklisted' );
            else
                update_term_meta( $term_id, 'next_check', time()-1 );

            return get_term( $term_id, $this->config_taxonomy_name );
        }

        function save_post_links( $post, $links, $force = false ) {
            if ( ! $force && $this->is_post_scanned( $post ) )
                return;

            $terms = [];

            foreach ( $links as $link ) {
                $term = $this->link_to_term( $link );

                if ( false === $term )
                    error_log( sprintf( __( 'RSLC: %s: link_to_term( $link ) failed.', 'TD'), site_url(), $link ) );
                else
                    $terms[] = $term->term_id;
            }

            $post = get_post( $post );
            wp_set_post_terms( $post->ID, $terms, $this->config_taxonomy_name );

            update_post_meta( $post->ID, '_rslc_last_check', time() );
        }


        function get_link_posts( $term ) {
            if ( is_a( $term, 'WP_Term' ) )
                $term = $term->term_id;

            $args = [
                'post_type' => $this->get_post_types(),
                'post_status' => $this->get_option( 'post_statuses' ),
                'numberposts' => -1,
                'tax_query' => [
                    [
                        'taxonomy' => $this->config_taxonomy_name,
                        'field' => 'term_id',
                        'terms' => $term ,
                    ]
                ],
            ];

            return get_posts( $args );

        }

        // Get all meta for term
        function get_term_meta( $term, $key = NULL ) {
            if ( is_a( $term, 'WP_Term' ) )
                $term = $term->term_id;

            if ( isset( $key ) )
                return get_term_meta( $term, $key, true );

            $meta = get_term_meta( $term );
            foreach ( array_keys( $meta ) as $key ) {
                if ( is_array( $meta[$key] ) && 1 == count( $meta[$key] ) )
                    $meta[$key] = $meta[$key][0];
            }

            return $meta;
        }

        // Update or delete as needed
        function update_term_meta( $term, $key, $value ) {
            if ( is_a( $term, 'WP_Term' ) )
                $term = $term->term_id;

            if ( $value )
                update_term_meta( $term, $key, $value );
            else
                delete_term_meta( $term, $key );
        }


        /************************************************************************
         *
         *	Blacklisting and user dismissale of links
         *
         ************************************************************************/

        protected $blacklist = NULL;

        function url_is_blacklisted( $url ) {
            if ( NULL === $this->blacklist ) {
                $this->blacklist = array_filter(
                    array_map(
                        fn( $line ) => trim( $line ),
                        explode( "\n", $this->get_option( 'link_blacklist_text' ) )
                    ),
                    fn( $line ) => !empty( $line )
                );
            }

            foreach ( $this->blacklist as $block ) {
                if ( str_contains( $url, $block ) )
                    return true;
            }

            return false;
        }

        // Dismiss or blacklist a link
        function dismiss_link( $link, $reason = 'user' ) {
            $this->update_term_meta( $link, 'dismissed', $reason );
            $this->update_term_meta( $link, 'next_check', false );
        }

        function undismiss_link( $link ) {
            $this->update_term_meta( $link, 'dismissed', false );
        }

        /************************************************************************
         *
         *	Lookup different types of links
         *
         ************************************************************************/

        function link_group_labels() {
            return [
                'all' => __( 'All links', 'rs-link-checker' ),
                'unchecked' => __( 'Unchecked', 'rs-link-checker' ),
                'ok' => __( 'Good', 'rs-link-checker' ),
                'redirect' => __( 'Redirecting', 'rs-link-checker' ),
                'broken' => __( 'Broken', 'rs-link-checker' ),
                'dismissed' => __( 'Dismissed', 'rs-link-checker' ),
                'blacklisted' => __( 'Blacklisted', 'rs-link-checker' ),
            ];
        }

        // All links
        function get_link_list( ) {
            $args = [
                'taxonomy' => $this->config_taxonomy_name,
                'fields' => 'all',
            ];

            return get_terms( $args );
        }

        function get_links_filtered( $filter = NULL ) {
            $args = [
                'taxonomy' => $this->config_taxonomy_name,
                'fields' => 'all',
            ];

            if ( empty( $filter ) )
                $filter = 'all';

            $args = apply_filters( "rslc_get_links_{$filter}", $args );

            return get_terms( $args );
        }

        function do_rslc_get_links_dismissed_filter( $args ) {
            $args['meta_key'] = 'dismissed';
            $args['meta_value'] = 'user';
            return $args;
        }

        function do_rslc_get_links_blacklisted_filter( $args ) {
            $args['meta_key'] = 'dismissed';
            $args['meta_value'] = 'blacklisted';
            return $args;
        }

        function do_rslc_get_links_ignored_filter( $args ) {
            $args['meta_key'] = 'dismissed';
            $args['meta_compare'] = 'EXISTS';
            return $args;
        }

        function do_rslc_get_links_unchecked_filter( $args ) {
            $args['meta_query'] = [
                [
                    'key' => 'status',
                    'compare' => 'NOT EXISTS',
                ],
                [
                    'key' => 'dismissed',
                    'compare' => 'NOT EXISTS',
                ],
            ];
            return $args;
        }

        function do_rslc_get_links_ok_filter( $args ) {
            $args['meta_query'] = [
                [
                    'key' => 'status',
                    'value' => [ 200, 299 ],
                    'compare' => 'BETWEEN',
                    'type' => 'NUMERIC',
                ],
                [
                    'key' => 'dismissed',
                    'compare' => 'NOT EXISTS',
                ],
            ];
            return $args;
        }

        function do_rslc_get_links_redirect_filter( $args ) {
            $args['meta_query'] = [
                [
                    'key' => 'status',
                    'value' => [ 300, 399 ],
                    'compare' => 'BETWEEN',
                    'type' => 'NUMERIC',
                ],
                [
                    'key' => 'dismissed',
                    'compare' => 'NOT EXISTS',
                ],
            ];
            return $args;
        }

        function do_rslc_get_links_broken_filter( $args ) {
            $args['meta_query'] = [
                [
                    'key' => 'status',
                    'value' => [ 400, 999 ],
                    'compare' => 'BETWEEN',
                    'type' => 'NUMERIC',
                ],
                [
                    'key' => 'dismissed',
                    'compare' => 'NOT EXISTS',
                ],
            ];
            return $args;
        }

        // For purging stale links
        function do_rslc_get_links_stale_filter( $args ) {
            $args['orderby'] = 'count';
            $args['order'] = 'asc';
            $args['hide_empty'] = false;
            return $args;
        }

        /************************************************************************
         *
         *	Post scanner methods
         *
         ************************************************************************/

        // Check if $post needs to be scanned
        function is_post_scanned( $post ) {
            $last_check = $post->_rslc_last_check;
            if ( ! $last_check )
                return false;

            $mtime = get_post_datetime( $post, 'modified' )->getTimeStamp();
            return ( $last_check >= $mtime );
        }

        // Do the hard work in a shutdown callback
        // Term creation can be slow
        function post_scanner_shutdown_callback( $post ) {
            $this->save_post_links( $post, $this->get_post_links( $post ) );
        }

        // Check status transitions and schedule update if needed
        function do_transition_post_status_action_update_links( $new, $old, $post ) {
            if ( ! in_array( $post->post_type, $this->get_post_types() ) )
                return;

            $statuses = $this->get_option( 'post_statuses', $this->config['post_statuses'] );
            if ( in_array( $new, $statuses ) )
                if ( ! $this->is_post_scanned( $post ) ) {
                    add_action( 'shutdown', fn() => $this->post_scanner_shutdown_callback( $post ) );
                }
            else {
                if ( in_array( $old, $statuses ) )
                    $this->save_post_links( $post, [] );
            }
        }

        // Find posts never scanned
        function get_post_queue_never( $numberposts = 0, $fields = '' ) {
            if ( ! $numberposts )
                $numberposts = $this->get_option( 'post_batch_size' );

            $args = [
                'post_type' => $this->get_post_types(),
                'post_status' => $this->get_option( 'post_statuses' ),
                'numberposts' => $numberposts,
                'fields' => $fields,
                'meta_query' => [
                    [
                        'key' => '_rslc_last_check',
                        'compare' => 'NOT EXISTS',
                    ]
                ],
            ];

            return get_posts( $args );
        }

        // Find posts which need a rescan
        function get_post_queue_old( $numberposts = 0, $fields = '' ) {
            if ( ! $numberposts )
                $numberposts = $this->get_option( 'post_batch_size' );

            $args = [
                'post_type' => $this->get_post_types(),
                'post_status' => $this->get_option( 'post_statuses' ),
                'numberposts' => -1,
                'fields' => $fields,
                'meta_key' => '_rslc_last_check',
                'meta_compare' => 'EXISTS',
            ];

            $posts = get_posts( $args );

            if ( empty( $posts ) ) {
                return [];
            }

            $rescan = array_filter( $posts,
                                    function( $post ) {
                                        $post = get_post( $post );
                                        $last_check = intval( $post->_rslc_last_check );
                                        if ( ! $last_check ) return true;

                                        $mtime = get_post_datetime( $post, 'modified' )->getTimeStamp();
                                        return ( $last_check < $mtime );
                                    }
            );

            return ( $numberposts > 0 ? array_splice( $rescan, 0, $numberposts ) : $rescan );
        }


        // Find posts which need a rescan
        function get_post_queue( $numberposts = 0, $fields = '' ) {
            if ( ! $numberposts )
                $numberposts = $this->get_option( 'post_batch_size' );

            $queue = $this->get_post_queue_never( $numberposts, $fields );

            if ( $numberposts > 0 ) {
                $numberposts = ( $numberposts >= count($queue) ? $numberposts - count( $queue ) : 0 );
            }
            if ( 0 != $numberposts ) {
                $rescan = $this->get_post_queue_old( $numberposts, $fields );

                if ( ! empty( $rescan ) )
                    $queue = array_merge( $queue, $rescan );
            }

            return $queue;
        }

        function post_scan_batch( $callback = NULL, $posts = [] ) {
            if ( empty( $posts ) )
                $posts = $this->get_post_queue( );

            foreach ( $posts as $post ) {
                $links = $this->get_post_links( $post );
                $this->save_post_links( $post, $links );

                if ( $callback )
                    call_user_func( $callback, $post, $links );
            }
        }


        function reset_posts() {
            $args = [
                'post_type' => $this->get_post_types(),
                'post_status' => $this->get_option( 'post_statuses' ),
                'numberposts' => -1,
                'fields' => 'ids',
                'meta_query' => [
                    [
                        'key' => '_rslc_last_check',
                        'compare' => 'EXISTS',
                    ]
                ],
            ];

            $posts = get_posts( $args );

            foreach ( $posts as $post ) {
                delete_post_meta( $post, '_rslc_last_check' );
            }
        }

        function post_stats() {
            return [
                'never' => count( $this->get_post_queue_never( -1, 'ids' ) ),
                'old' => count( $this->get_post_queue_old( -1, 'ids' ) ),
            ];
        }

        function post_scan_status() {
            $stats = $this->post_stats();
            $total = array_sum( array_values( $stats ) );

            $msg = __( "The post scan queue is empty", 'rs-link-checker' );
            $args = [];

            if ( $total ) {
                if ( 0 ==  $stats['never'] ) {
                    $msg = __( 'The post scan queue has %d posts to rescan', 'rs-link-checker' );
                    $args[] = $stats['old'];
                }
                elseif ( 0 ==  $stats['old'] ) {
                    $msg = __( 'The post scan queue has %d new posts to scan', 'rs-link-checker' );
                    $args[] = $stats['never'];
                }
                else {
                    $msg = __( 'The post scan queue has %d new posts and %d old posts to scan', 'rs-link-checker' );
                    $args[] = $stats['never'];
                    $args[] = $stats['old'];
                }


                $nonce = wp_create_nonce( 'rslc-scan-posts-nonce' );
                $return = urlencode( site_url( $_SERVER['REQUEST_URI'] ) );

                $url = $this->build_admin_url( 'admin-post.php', [
                    'action' => "rslc_scan_posts",
                    '_wpnonce' => $nonce,
                    'return' => $return,
                ] );

                $msg .= ' - <a href="%s">%s</a>';
                $args[] = $url;
                $args[] = __( 'do it now!', 'rs-link-checker' );
            }

            return vsprintf( $msg, $args );
        }

        /************************************************************************
         *
         *	Post parser methods
         *
         ************************************************************************/

        protected $tags_attrs = [
            'A' => 'href',
            'EMBED' => 'src',
            'IFRAME' => 'src',
            'IMG' => 'src',
            'SCRIPT' => 'src',
            'SOURCE' => 'src',
            'AUDIO' => 'src',
            'VIDEO' => 'src',
        ];

        function get_post_raw_links( $post ) {
            $post = get_post( $post );

            if ( empty( $post ) )
                return NULL;

            $input = $post->post_content;
            $links = [];

            // Allow custom modifications - e.g., footnotes
            $input = apply_filters( 'rslc_pre_content_parse', $input, $post );

            $tags = new \WP_HTML_Tag_Processor( $input );
            while ( $tags->next_tag() ) {
                $tag = $tags->get_tag();

                if ( array_key_exists( $tag, $this->tags_attrs ) ) {
                    $link = $tags->get_attribute( $this->tags_attrs[ $tag ] );
                    if ( $link ) {
                        $rel = $tags->get_attribute( 'rel' );
                        if ( !$rel || ! str_contains( $rel, 'nofollow' ) )
                            $links[] = $link;
                    }
                }
            }

            return $links;
        }

        protected function cleanup_url( $url ) {
            $parts = wp_parse_url( $url );

            if ( empty( $parts['host'] ) ) {
                // skip self links to internal anchors -- like href="#xxx"
                if ( empty( $parts['path'] ) && !empty( $parts['fragment']) )
                    return;

                $url = \WP_Http::make_absolute_url( $url, get_option( 'siteurl' ) );
                $parts = wp_parse_url( $url );
            }

            if ( ! in_array( $parts['scheme'], [ 'https', 'http' ] ) )
                return;

            return
                (isset($parts['scheme']) ? "{$parts['scheme']}:" : '')
                . ((isset($parts['user']) || isset($parts['host'])) ? '//' : '')
                . (isset($parts['user']) ? "{$parts['user']}" : '')
                . (isset($parts['pass']) ? ":{$parts['pass']}" : '')
                . (isset($parts['user']) ? '@' : '')
                . (isset($parts['host']) ? "{$parts['host']}" : '')
                . (isset($parts['port']) ? ":{$parts['port']}" : '')
                . (isset($parts['path']) ? "{$parts['path']}" : '')
                . (isset($parts['query']) ? "?{$parts['query']}" : '')
                ;
        }

        function get_post_filtered_links( $post ) {
            return array_filter( array_map( fn( $link ) => $this->cleanup_url( $link ),
                                            $this->get_post_raw_links( $post ) ),
                                 fn( $link ) => ! empty( $link )
            );
        }

        function get_post_links( $post ) {
            return array_unique( $this->get_post_filtered_links( $post ) );
        }

        function get_post_aggregate_links( $post ) {
            $links = $this->get_post_filtered_links( $post );

            $aggrlinks = [];

            foreach ( $links as $link ) {
                if ( array_key_exists( $link, $aggrlinks ) )
                    $aggrlinks[$link]++;
                else
                    $aggrlinks[$link] = 1;
            }

            return $aggrlinks;
        }


        /************************************************************************
         *
         *	Rescan posts with link
         *
         ************************************************************************/

        function rescan_link_posts( $link ) {
            $posts = $this->get_link_posts( $link );

            foreach ( $posts as $post ) {
                $links = $this->get_post_links( $post );

                if ( $this->get_option( 'logging' ) )
                    error_log( sprintf( __( 'RSLC %s: post %d has %d links', 'rs-link-checker' ),
                                        site_url(), $post->ID, count($links) )
                    );

                $this->save_post_links( $post, $links, true );
            }
        }

        /************************************************************************
         *
         *	Modify links in posts
         *
         ************************************************************************/

        function update_link_posts( $link, $new = NULL ) {
            $posts = $this->get_link_posts( $link );
            if ( empty( $posts ) ) return;

            $meta = $this->get_term_meta( $link );

            $old = $meta['url'];

            if ( empty( $new ) )
                $new = $meta['redirect'];

            if ( empty( $new ) ) return;

            foreach ( $posts as $post ) {
                $this->update_post_link( $post, $old, $new );
            }
        }

        function update_post_link( $post, $old, $new ) {
            if ( $old == $new )
                return;

            $updated_content = $this->update_html_link( $post->post_content, $old, $new, get_permalink( $post ) );
            $changed = ( $updated_content != $post->post_content );

            // Allow other modifications, e.g., footnotes in meta fields
            $changed = apply_filters( 'rslc_update_post_link', $changed, $post, $old, $new );

            if ( $changed ) {
                wp_update_post( [
                    'ID' => $post->ID,
                    'post_content' => $updated_content,
                ] );
            }
        }

        // Change link $old to $new in $input HTML - uses the same
        // tag/attribute table as the parser
        function update_html_link( $input, $old, $new, $abs = NULL ) {
            $tags = new \WP_HTML_Tag_Processor( $input );

            while ( $tags->next_tag() ) {
                $tag = $tags->get_tag();

                if ( array_key_exists( $tag, $this->tags_attrs ) ) {
                    $link = $tags->get_attribute( $this->tags_attrs[ $tag ] );

                    // Ignore links which are just a #fragment
                    // WP makes such links for footnotes
                    if ( $link && !str_starts_with( $link, '#' ) ) {
                        if ( $abs )
                            $link = \WP_Http::make_absolute_url( $link, $abs );

                        // Also replace links with a fragment
                        $fragment = wp_parse_url( $link, PHP_URL_FRAGMENT );
                        if ( empty( $fragment ) ) {
                            if ( $link == $old )
                                $tags->set_attribute(  $this->tags_attrs[ $tag ], $new );
                        } else {
                            if ( $link == "$old#$fragment" )
                                $tags->set_attribute(  $this->tags_attrs[ $tag ], "$new#$fragment" );
                        }
                    }
                }
            }

            return $tags->get_updated_html();
        }


        /************************************************************************
         *
         *	Add rel=nofollow for link in posts
         *
         ************************************************************************/

        function update_link_posts_nofollow( $link ) {
            $posts = $this->get_link_posts( $link );
            if ( empty( $posts ) ) return;

            $meta = $this->get_term_meta( $link );
            $url = $meta['url'];

            foreach ( $posts as $post ) {
                $this->update_post_link_nofollow( $post, $url );
            }
        }

        function update_post_link_nofollow( $post, $url ) {
            $updated_content = $this->update_html_link_nofollow( $post->post_content, $url, get_permalink( $post ) );
            $changed = ( $updated_content != $post->post_content );

            // Allow other modifications, e.g., footnotes in meta fields
            $changed = apply_filters( 'rslc_update_post_link_nofollow', $changed, $post, $url );

            if ( $changed ) {
                wp_update_post( [
                    'ID' => $post->ID,
                    'post_content' => $updated_content,
                ] );
            }
        }

        function update_html_link_nofollow( $input, $url, $new, $abs = NULL ) {
            $tags = new \WP_HTML_Tag_Processor( $input );

            while ( $tags->next_tag( 'a' ) ) {
                $link = $tags->get_attribute( 'href' );

                // Ignore links which are just a #fragment
                // WP makes such links for footnotes
                if ( $link && !str_starts_with( $link, '#' ) ) {
                    if ( $abs )
                        $link = \WP_Http::make_absolute_url( $link, $abs );

                    // Also replace links with a fragment
                    $fragment = wp_parse_url( $link, PHP_URL_FRAGMENT );
                    if ( empty( $fragment ) ) {
                        if ( $link == $url )
                            $tags->set_attribute( 'rel', 'nofollow' );
                    } else {
                        if ( $link == "$url#$fragment" )
                            $tags->set_attribute(  'rel', 'nofollow' );
                    }
                }
            }

            return $tags->get_updated_html();
        }

        /************************************************************************
         *
         * Footnote handling
         *
         * - Add the footnote content to the parser input.
         * - Update links in the JSON content of the footnote meta field.
         *
         ************************************************************************/

        function do_rslc_pre_content_parse_filter( $input, $post ) {
            $footnotes = $this->footnotes_parse_meta( $post );

            foreach ( $footnotes as $footnote ) {
                if ( !empty( $footnote['content'] ) )
                    $input .= "\n<div>" . $footnote['content'] . "</div>\n";
            }

            return $input;
        }

        function do_rslc_update_post_link_filter( $changed, $post, $old, $new ) {
            $footnotes = $this->footnotes_parse_meta( $post );

            $output = [];
            $footnote_changed = false;
            foreach ( $footnotes as $footnote ) {
                if ( !empty( $footnote['content'] ) ) {
                    $new_content = $this->update_html_link( $footnote['content'], $old, $new, get_permalink( $post ) );

                    if ( $new_content != $footnote['content'] ) {
                        $footnote['content'] = $new_content;
                        $footnote_changed = true;
                    }
                }
                $output[] = $footnote;
            }

            if ( $footnote_changed ) {
                $new_meta = wp_json_encode( $output );
                update_post_meta( $post->ID, 'footnotes', wp_slash( $new_meta ) );
            }

            return $changed || $footnote_changed;
        }

        function footnotes_parse_meta( $post ) {
            $meta = $post->footnotes;
            if ( empty( $meta ) )
                return [];

            try {
                return json_decode( $meta, true );
            }
            catch ( ValueException ) {
                error_log( sprintf( __( 'RSLC %s: post %d: failed to parse footnotes', 'rs-link-checker' ),
                                    site_url(), $post->ID )
                );
            }
            return [];
        }

        /************************************************************************
         *
         *	Link checker methods
         *
         ************************************************************************/

        function get_link_queue( $count = 0 ) {
            $args = [
                'taxonomy' => $this->config_taxonomy_name,
                'number' => ( $count == 0 ?  $this->get_option( 'link_batch_size' ) :  $count ),
                'fields' => 'all',
                'hide_empty' => false,
                'orderby' => 'meta_value',
                'meta_key' => 'next_check',
                'meta_value' => time(),
                'meta_compare' => '<=',
                'meta_type' => 'NUMERIC',
            ];

            return get_terms( $args );
        }

        function check_link( $term ) {
            $term = $this->get_link( $term );
            $term_id = $term->term_id;

            if ( 0 === $term->count ) {
                if ( $this->get_option( 'logging' ) )
                    error_log( sprintf( __( 'RSLC %s: deleting link %s', 'rs-link-checker' ),
                                        site_url(), $this->get_term_meta( $term_id, 'url' ) ) );
                wp_delete_term( $term_id, $this->config_taxonomy_name );
                return;
            }

            $link = $this->get_term_meta( $term_id, 'url' );
            if ( ! $link ) return;

            if ( $this->url_is_blacklisted( $link ) ) {
                $this->dismiss_link( $term, 'blacklisted' );
                return;
            }

            $status = NULL;
            $message = NULL;
            $redirect = NULL;

            $ua = $this->get_option( 'user_agent', $this->config['user_agent'] );

            $response = wp_remote_get( $link, [
                'sslverify' => false,
                'redirection' => 0,
                'timeout' => $this->get_option('timeout', 5),
                'limit_response_size' => 16 * 1024,
                'user-agent' => $ua,
                'headers' => [
                    'Referer: ' . site_url(),
                ],
            ] );
            if ( is_wp_error( $response ) ) {
                $status = 999;
                $message = $response->get_error_message();
            } else {
                $status = (int)$response['response']['code'];
                $message = $response['response']['message'];

                if ( isset( $response['headers']['location'] ) ) {
                    $redirect = $response['headers']['location'];
                    $redirect = \WP_Http::make_absolute_url( $redirect, $link );
                }
            }

            $this->update_term_meta( $term_id, 'next_check', time() + $this->get_option( 'link_interval' ) );

            $this->update_term_meta( $term_id, 'status', $status );
            $this->update_term_meta( $term_id, 'message', $message );
            $this->update_term_meta( $term_id, 'redirect', $redirect );
            $this->update_term_meta( $term_id, 'response', $response );

            if ( $status < 300 )
                $this->undismiss_link( $term_id );
        }

        function link_check_batch( $callback = NULL ) {
            $terms = $this->get_link_queue();

            foreach ( $terms as $term ) {
                $this->check_link( $term );

                if ( $callback )
                    call_user_func( $callback, $term->term_id );
            }
        }

        function link_group_counts() {
            $counts = array_fill_keys( array_keys( $this->link_group_labels() ), 0 );

            $links = $this->get_link_list();
            foreach ( $links as $link ) {
                $dismissed = $this->get_term_meta( $link->term_id, 'dismissed' );

                if ( $dismissed ) {
                    if ( 'blacklisted' == $dismissed )
                        $counts['blacklisted']++;
                    else
                        $counts['dismissed']++;

                } else {
                    $status = intval( $this->get_term_meta( $link->term_id, 'status' ) );

                    if ( $status >= 400 )
                        $counts['broken']++;
                    elseif ( $status >= 300 )
                        $counts['redirect']++;
                    elseif ( $status >= 200 )
                        $counts['ok']++;
                    else
                        $counts['unchecked']++;
                }
            }

            $counts['all'] = array_sum( $counts );

            return $counts;
        }

        function get_link( $link ) {
            if ( ! is_a( $link, 'WP_Term' ) )
                $link = get_term( (int)$link, $this->config_taxonomy_name );
            return $link;
        }

        function link_is_unchecked( $link ) {
            $link = $this->get_link( $link );
            if ( !$link ) return false;
            $status = $this->get_term_meta( $link, 'status' );
            return !$status;
        }

        function link_is_checked_good( $link ) {
            $link = $this->get_link( $link );
            if ( !$link ) return false;
            $status = $this->get_term_meta( $link, 'status' );
            return ( $status >= 200 && $status < 300 );
        }

        function link_is_redirect( $link ) {
            $link = $this->get_link( $link );
            if ( !$link ) return false;
            $status = $this->get_term_meta( $link->term_id, 'status' );
            $redirect = $this->get_term_meta( $link->term_id, 'redirect' );
            return ( !empty( $redirect ) && $status >= 300 && $status < 400 );
        }

        function link_is_broken( $link ) {
            $link = $this->get_link( $link );
            if ( !$link ) return false;
            $status = $this->get_term_meta( $link, 'status' );
            return ( $status >= 400 );
        }

        function link_is_dismissed( $link ) {
            $link = $this->get_link( $link );
            if ( !$link ) return false;
            $dismissed = $this->get_term_meta( $link, 'dismissed' );
            return ( $dismissed && ( 'user' == $dismissed || true === $dismissed ) );
        }

        function link_is_blacklisted( $link ) {
            $link = $this->get_link( $link );
            if ( !$link ) return false;
            $dismissed = $this->get_term_meta( $link, 'dismissed' );
            return ( $dismissed && 'blacklisted' == $dismissed );
        }


        /************************************************************************
         *
         *	Mail list of broken links
         *
         ************************************************************************/

        function send_mail_with_broken_links( $broken = NULL ) {
            if ( $broken === NULL )
                $broken = $this->get_links_filtered( 'broken' );

            if ( empty( $broken ) )
                return;

            $to = get_option( 'admin_email' );
            $subject = sprintf( __( '%s: %d broken links', 'rs-link-checker' ),
                                get_option( 'blogname' ),
                                count( $broken )
            );

            // Get all term meta data and sort by url
            $broken_meta = array_map( fn( $term ) => $this->get_term_meta( $term ), $broken );
            usort( $broken_meta, fn( $a, $b ) => $a['url'] <=> $b['url'] );

            // Split broken links by status
            $broken_by_status = [];
            foreach ( $broken_meta as $meta ) {
                if ( $meta['status'] == 999 ) {
                    if ( preg_match( '/^cURL error (\d+): (.+?)(: .*)?$/', $meta['message'], $match ) ) {
                        $meta['status'] = 900 + intval( $match[1] );
                        $meta['message'] = $match[2];
                    }
                }
                $broken_by_status[ $meta['status'] ][] = $meta;
            }
            ksort( $broken_by_status, SORT_NUMERIC );

            // Generate email message body

            ob_start();

            printf( "Website '%s' has %d broken links.\n\n", get_option( 'blogname' ), count( $broken ) );

            foreach ( array_keys( $broken_by_status ) as $status ) {
                printf( "%s %s (%d links)\n", $status, $broken_by_status[$status][0]['message'], count( $broken_by_status[$status] ) );

                foreach ( $broken_by_status[$status] as $meta )
                    printf( "- %s\n", $meta['url'] );

                print( "\n" );
            }

            printf( "Fix the errors here: %s\n\n", $this->management_page_link() );
            printf( "Server IP addr: %s\n", $_SERVER['SERVER_ADDR'] ?? __( 'not set', 'rs-link-checker' ) );

            $message = ob_get_clean();

            wp_mail( $to, $subject, $message );
        }

        /************************************************************************
         *
         *	Scheduler
         *
         ************************************************************************/

        protected $cron_action = 'rs_link_checker';

        function do_rs_link_checker_action() {
            $callback = NULL;

            if ( $this->get_option( 'logging' ) )  {
                $callback = function( $post, $links ) {
                    error_log( sprintf( __( 'RSLC %s: post %d has %d links', 'rs-link-checker' ),
                                        site_url(), $post->ID, count($links) )
                    );
                };
            }

            $this->post_scan_batch( $callback );

            if ( $this->get_option( 'logging' ) )  {
                $callback = function( $term_id ) {
                    $term = get_term( $term_id, $this->config_taxonomy_name );
                    if ( $term ) {
                        $meta = $this->get_term_meta( $term );
                        error_log( sprintf( __( 'RSLC: %s: link %d / %s: %d %s - %s', 'rs-link-checker' ),
                                            site_url(), $term_id, $term->slug, $meta['status'], $meta['message'], $meta['url'] )
                        );
                    }
                };

            }

            // Experiemental - sometimes CPT generates 404 on everything
            flush_rewrite_rules();


            $prev_broken = count( $this->get_links_filtered( 'broken' ) ?: [] );

            $this->link_check_batch( $callback );

            $broken = $this->get_links_filtered( 'broken' );

            // Nothing broken - no mail
            if ( empty( $broken ) )
                return;

            // Same or less as before - no mail
            if ( count( $broken ) <= $prev_broken )
                return;

            // Something's broken and more than before, so send mail
            $this->send_mail_with_broken_links( $broken );
        }

        function do_init_action_schedule_callback() {
            register_deactivation_hook(  __FILE__, function () {
                wp_clear_scheduled_hook( $this->cron_action );
            } );

            $schedule = $this->get_option( 'schedule' );
            if ( isset( $schedule ) and $schedule == 'none' )
                $schedule = NULL;

            if ( $schedule ) {
                $schedules = wp_get_schedules();
                if ( !isset( $schedules[$schedule] ) )
                    $schedule = NULL;
            }

            // Next scheduled event
            $event = wp_get_scheduled_event( $this->cron_action );

            if ( $schedule ) {
                if ( $event === false ) {
                    // Not scheduled but should be
                    wp_schedule_event( time(), $schedule, $this->cron_action );
                }
                elseif ( $event->schedule != $schedule ) {
                    // Scheduled but not correctly
                    wp_clear_scheduled_hook( $this->cron_action );

                    $when = $event->timestamp;
                    if ( $when > time() + $schedules[$schedule]['interval'] )
                        $when = time() + $schedules[$schedule]['interval'];

                    wp_schedule_event( $when, $schedule, $this->cron_action );
                }
            } else {
                // Shouldn't be scheduled
                if ( $event )
                    wp_clear_scheduled_hook( $this->cron_action );
            }
        }

        // Add 15 and 30 minutes intervals
        function do_cron_schedules_filter_add_cron_intervals( $schedules ) {
            $extras = [
                'fifteen_minutes' => [
                    'interval' => 15 * 60,
                    'display'  => esc_html__( 'Every 15 minutes', 'rs-link-checker' ),
                ],
                'half-hour' => [
                    'interval' => 30 * 60,
                    'display'  => esc_html__( 'Every half hour', 'rs-link-checker' ),
                ],
            ];

            foreach ( $extras as $key => $extra ) {
                if ( empty( array_filter( $schedules, fn( $s ) => ( $s['interval'] == $extra['interval'] ) ) ) )
                    $schedules[ $key ] = $extra;
            }

            return $schedules;
        }


        function scheduler_status() {
            $next = wp_next_scheduled( $this->cron_action );

            if ( ! $next )
                return __( 'Scans and checks not scheduled within WordPress.', 'rs-link-checker' );

            $format = get_option( 'date_format' ) . ' @ ' . get_option( 'time_format' );
            $when = wp_date( $format , $next );
            return sprintf( __( 'Next scheduled scan/check at %s', 'rs-link-checker' ), $when );
        }

        /************************************************************************
         *
         *	Cleanup and bookkeeping
         *
         ************************************************************************/


        // TODO: remove terms with count == 0











        /************************************************************************
         *
         *	Reports
         *
         ************************************************************************/




        /************************************************************************
         *
         *	Lists of Links - using class Link_Checker_Table
         *
         ************************************************************************/

        public $management_page = NULL;
        public $management_page_name = 'link_checker_page';

        function do_admin_menu_action_tools_page() {
            $this->management_page = add_management_page(
                __('Link Checker', 'rs-link-checker' ),
                __('RS Link Checker', 'rs-link-checker' ),
                'edit_posts',
                $this->management_page_name,
                [ $this, 'management_page_render' ]
            );

            add_action( "load-{$this->management_page}", [ $this, 'load_managament_page_hook' ] );
        }

        function management_page_link( $filter = NULL, $copy = true ) {
            $args = [ 'page' => $this->management_page_name ];

            if ( isset( $filter ) )
                $args['filter'] = $filter;

            if ( $copy ) {
                foreach ( [ 'order', 'orderby', 'filter', 's' ] as $param ) {
                    if ( ! array_key_exists( $param, $args ) && isset( $_REQUEST[$param] ) )
                        $args[$param] = $_REQUEST[$param];
                }
            }

            return $this->build_admin_url( 'tools.php', $args );
        }

        function management_page_render() {
            printf( '<div class="wrap"><h2>%s</h2>', __( 'RS Link Checker', 'rs-link-checker' ) );

            printf( '<form method="get">' );
            printf( '<input type="hidden" name="page" value="%s" />', esc_attr( $this->management_page_name ) );

            $table = new Link_Checker_Table( $this, $this->management_page );

            $table->do_bulk_actions();

            printf( '<p>%s.</p>', $this->post_scan_status() );
            printf( '<p>%s.</p>', $this->scheduler_status() );

            $table->prepare_items();
            $table->output_hidden_fields();
            $table->views();
            $table->search_box( __( 'Search', 'rs-link-checker' ), $this->management_page . '-search' );
            $table->display();

            echo '</form>';
            echo '</div>';
        }


        // Row actions for Link_Checker_Table

        function get_row_actions() {
            return [
                'edit' => [
                    'check' => fn($link) => true,
                    'label' => __( 'Edit URL', 'rs-link-checker' ),
                ],
                'recheck' => [
                    'check' => fn($link) => true,
                    'label' => __( 'Recheck', 'rs-link-checker' ),
                ],
                'fixredirect' => [
                    'check' => fn($link) => $this->get_term_meta( $link->term_id, 'redirect'),
                    'label' => __( 'Update', 'rs-link-checker' ),
                ],
                'nofollow' => [
                    'check' => fn($link) => ( (int)$this->get_term_meta( $link, 'status') ?: 300 ) >= 300,
                    'label' => __( 'Nofollow', 'rs-link-checker' ),
                ],
                'rescan' => [
                    'check' => fn($link) => $link->count,
                    'label' => __( 'Rescan', 'rs-link-checker' ),
                ],
                'dismiss' => [
                    'check' => fn($link) => ! $this->get_term_meta( $link, 'dismissed') && (int)$this->get_term_meta( $link, 'status') >= 300,
                    'label' => __( 'Dismiss', 'rs-link-checker' ),
                ],
                'undismiss' => [
                    'check' => fn($link) => ( $this->get_term_meta( $link, 'dismissed') ?? '' ) == 'user',
                    'label' => __( 'UnDismiss', 'rs-link-checker' ),
                ],
            ];
        }

        // Single Link_Checker_Table action helpers

        function admin_post_action_check_nonce( $name = 'rslc-row-actions-nonce' ) {
            if ( !( isset( $_REQUEST['_wpnonce'] ) && wp_verify_nonce( $_REQUEST['_wpnonce'], $name ) ) )
                wp_die( __( 'Security check', 'rs-link-checker' ) );
        }

        function admin_post_action_return() {
            if ( ! isset( $_REQUEST['return'] ) )
                wp_die( __( 'No return parameter specified', 'rs-link-checker' ) );
            $return = urldecode( $_REQUEST['return'] );
            wp_redirect( $return );
            exit;
        }

        function admin_post_action_get_link() {
            if ( isset( $_REQUEST['link'] ) ) {
                $term_id = intval( $_REQUEST['link'] );

                if ( $term_id )
                    return get_term( $term_id, $this->config_taxonomy_name );
            }

            die( __( 'No link parameter specified', 'rs-link-checker' ) );
        }

        // Single Link_Checker_Table 'scan posts' link

        function do_admin_post_rslc_scan_posts_action() {
            $this->admin_post_action_check_nonce( 'rslc-scan-posts-nonce');
            $this->post_scan_batch(
                function( $post, $links ) {
                    error_log( sprintf( __( 'RSLC %s: post %d has %d links', 'rs-link-checker' ),
                                        site_url(), $post->ID, count($links) )
                    ); } );
            $this->admin_post_action_return();
        }

        // Single Link_Checker_Table actions handlers

        function do_admin_post_rslc_dismiss_action() {
            $this->admin_post_action_check_nonce();
            $link = $this->admin_post_action_get_link();

            $this->dismiss_link( $link );

            $this->admin_post_action_return();
        }

        function do_admin_post_rslc_undismiss_action() {
            $this->admin_post_action_check_nonce();
            $link = $this->admin_post_action_get_link();

            $this->undismiss_link( $link );

            $this->admin_post_action_return();
        }

        function do_admin_post_rslc_edit_action() {
            $this->admin_post_action_check_nonce();
            $link = $this->admin_post_action_get_link();

            if ( isset( $_REQUEST['url'] ) ) {
                $newurl = $_REQUEST['url'];
                if ( $newurl )
                    $this->update_link_posts( $link, $newurl );
            }
            // Do something

            $this->admin_post_action_return();
        }

        function do_admin_post_rslc_recheck_action() {
            $this->admin_post_action_check_nonce();
            $link = $this->admin_post_action_get_link();

            $this->check_link( $link );

            $this->admin_post_action_return();
        }

        function do_admin_post_rslc_fixredirect_action() {
            $this->admin_post_action_check_nonce();
            $link = $this->admin_post_action_get_link();

            if ( $this->link_is_redirect( $link ) )
                $this->update_link_posts( $link );

            $this->admin_post_action_return();
        }

        function do_admin_post_rslc_nofollow_action() {
            $this->admin_post_action_check_nonce();
            $link = $this->admin_post_action_get_link();

            $this->update_link_posts_nofollow( $link );

            $this->admin_post_action_return();
        }

        function do_admin_post_rslc_rescan_action() {
            $this->admin_post_action_check_nonce();
            $link = $this->admin_post_action_get_link();

            $this->rescan_link_posts( $link );

            $this->admin_post_action_return();
        }

        // Bulk Link_Checker_Link_Table actions

        function bulk_action_get_links() {
            if ( !isset( $_REQUEST['ids'] ) ) return [];
            if ( !is_array( $_REQUEST['ids'] ) ) return [];

            return array_map( fn($id) => intval( $id ), $_REQUEST['ids'] );
        }

        function do_link_checker_page_bulk_dismiss_action() {
            $ids = $this->bulk_action_get_links();

            foreach ( $ids as $id )
                $this->dismiss_link( $id );
        }

        function do_link_checker_page_bulk_undismiss_action() {
            $ids = $this->bulk_action_get_links();

            foreach ( $ids as $id )
                $this->undismiss_link( $id );
        }

        function do_link_checker_page_bulk_recheck_action() {
            $ids = $this->bulk_action_get_links();

            foreach ( $ids as $id ) {
                $this->check_link( $id );
            }
        }

        function do_link_checker_page_bulk_rescan_action() {
            $ids = $this->bulk_action_get_links();

            foreach ( $ids as $id ) {
                $this->rescan_link_posts( $id );
            }
        }


        function do_link_checker_page_bulk_fixredirect_action() {
            $ids = $this->bulk_action_get_links();

            foreach ( $ids as $id ) {
                $link = get_term( $id, $this->config_taxonomy_name );
                if ( $this->link_is_redirect( $link ) )
                    $this->update_link_posts( $link );
            }
        }

        function do_link_checker_page_bulk_nofollow_action() {
            $ids = $this->bulk_action_get_links();

            foreach ( $ids as $id ) {
                $link = get_term( $id, $this->config_taxonomy_name );
                $this->update_link_posts_nofollow( $link );
            }
        }

        // Set up the Screen Options

        function load_managament_page_hook() {
            $screen = get_current_screen();

            // get out of here if we are not on our settings page
            if( !is_object( $screen ) || $screen->id != $this->management_page )
                return;

            $args = array(
                'label' => __( 'Elements per page', 'rs-link-checker' ),
                'default' => 20,
                'option' => 'elements_per_page'
            );

            add_screen_option( 'per_page', $args );

            // This allows column selection - I have no idea why
            new Link_Checker_Table( $this, $this->management_page );
        }

        protected $set_screen_option_callback_hook = 'set-screen-option';

        function do_set_screen_option_filter( $status, $option, $value ) {
            return $value;
        }

        // Make the last columns narrower
        function do_admin_head_action_link_table_css() {
            echo '<style type="text/css">';
            echo ".{this->management_page} .column-status { width: 10% !important; overflow: hidden }";
            echo ".{this->management_page} .column-date { width: 14% !important; overflow: hidden; }";
            echo '</style>';
        }


        // Enqueue admin JS script

        function do_admin_enqueue_scripts_action_javascript_helpers ( $hook_suffix ) {
            if ( $hook_suffix != $this->management_page )
                return;

            $plugin_data = get_file_data( __FILE__ , [ 'Version' => 'Version' ] );
            $version = $plugin_data['Version'] ?? '0';

            wp_enqueue_script( 'rs-link-checker-javascript-helpers',
                               plugins_url( '/js/link-checker-helpers.js', __FILE__ ),
                               [ 'jquery' ],
                               $version,
                               [ 'in_footer' => true, ],
            );
        }





        function do_admin_notices_action_broken_links() {
            if ( isset( $_REQUEST['page'] ) && $_REQUEST['page'] == $this->management_page_name )
                return;

            $broken = $this->get_links_filtered( 'broken' );

            if ( !empty( $broken ) ) {
                echo '<div class="notice notice-warning is-dismissible"><p>';
                $text = _n( 'detected %d link with errors',
                            'detected %d links with errors',
                            count( $broken )
                );

                $text = sprintf( $text, count( $broken ) );
                printf( 'RS Link Checker: <a href="%s">%s</a>.',
                        $this->management_page_link( 'broken', false ),
                        $text
                );
                echo '</p></div>';
            }
        }







        /************************************************************************
         *
         *	Dashboard
         *
         ************************************************************************/


        function do_dashboard_glance_items_filter( $items ) {
            $broken = $this->get_links_filtered( 'broken' );
            if ( empty( $broken ) )
                return $items;

            $text = sprintf( _n( '%d broken link', '%d broken links', count( $broken ) ), count( $broken ) );

            $items[] = sprintf( '<a class="%s" href="%s">%s</a>',
                                "at-a-glance-broken-link-count",
                                esc_url( $this->management_page_link( 'broken', false ) ),
                                esc_html( $text )
            );

            return $items;
        }

        // Add the correct icons to the At a Glance dashboard widget
        function do_admin_head_action_at_a_glance_css() {
            echo '<style type="text/css">';
            echo '#dashboard_right_now li a.at-a-glance-broken-link-count::before { content: "225"; }';
            echo '</style>';
        }



        function do_wp_dashboard_setup_action_status_widget() {
            wp_add_dashboard_widget(
                'rs_link_checker_dashboard_widget',
                esc_html__( 'RS Link Checker status', 'rs-link-checker' ),
                [ $this, 'dashboard_widget_render' ]
            );
        }

        /**
         * Create the function to output the content of our Dashboard Widget.
         */
        function dashboard_widget_render() {
            // Post scan status
            printf( "<p>%s.</p>\n", $this->post_scan_status() );

            // Link check status
            $counts = $this->link_group_counts();
            $labels = $this->link_group_labels();

            $output = [];
            foreach ( $labels as $key => $label ) {
                if ( 0 == $counts[ $key ] )
                    continue;
                $output[] = sprintf( '<a href="%s">%s (%d)</a>',
                                     $this->management_page_link( $key ), $label, $counts[ $key ] );
            }
            printf( '<p>%s</p>', join( ' · ', $output ) );

            // Scheduler status
            printf( "<p>%s.</p>\n", esc_html( $this->scheduler_status() ) );
        }


        /************************************************************************
         *
         *	Options
         *
         ************************************************************************/

        function filter_option_list( $option, $valid_values, $default = [] ) {
            $input = $this->get_option( $option );
            $output = [];

            if ( ! empty( $input ) && is_array( $input ) ) {
                $output = array_filter( $input,  fn ( $v ) => !empty( $v ) && in_array( $v, $valid_values) );
            }

            if ( empty( $output ) )
                $output = $default ?? [];

            return array_values( $output );
        }



        /************************************************************************
         *
         *	Settings
         *
         ************************************************************************/

        function do_plugin_action_links___PLUGIN_BASENAME___action_management_page( $links ) {
            $url = $this->management_page_link();
            $text = __( 'Manage', 'TD' );
            $links[] = sprintf( '<a href="%s">%s</a>', esc_url( $url ), esc_html( $text ) );
            return $links;
        }

        function do_admin_menu_action() {
            $this->settings_add_menu(
                __( 'RS Link Checker', 'rs-link-checker' ),
                __( 'RS Link Checker', 'rs-link-checker' )
            );
        }

        function settings_define_sections_and_fields() {

            // Scheduling

            $section = $this->settings_add_section(
                'section_scheduling',
                __( 'Scheduling', 'rs-link-checker' ),
                __( 'Scheduling is about how often you want the plugin to scan posts for links and check if the found links work.', 'rs-link-checker' )
            );


            $schedule_menu = [ 'none' => __('None', 'rs-link-checker') ];
            foreach ( wp_get_schedules() as $k => $v )
                $schedule_menu[$k] = $v['display'];

            $this->settings_add_field(
                'schedule', $section,
                __( 'Scheduling', 'rs-link-checker' ),
                'settings_field_select_html',
                [
                    'values' => $schedule_menu,
                ]
            );



            // Post types

            $section = $this->settings_add_section(
                'section_posts',
                __( 'Post related settings', 'rs-link-checker' ),
                __( 'The plugin will search posts for links which should be checked. Select which post types you want the plugin to work on, and which post statuses. Posts are automatically scanned when saved, but check for posts in need of a rescan will be performed regularly.', 'rs-link-checker' )
            );

            $this->settings_build_post_types_menu(
                'post_types', $section,
                __( 'Post-types', 'rs-link-checker' ),
                [ 'public' => true ],
                [],
                [
                    'rs_link_checker_post_types',
                ]
            );


            $this->settings_add_field(
                'synced_patterns', $section,
                __( 'Also check links in synced patterns?', 'rs-link-checker' ),
                'settings_field_checkbox_html'
            );


            // Post status

            $values = [];
            $post_statuses = get_post_stati( [ 'internal' => false ], 'objects' );
            foreach ( $post_statuses as $post_status ) {
                $values[ $post_status->name ] = $post_status->label;
            }

            $this->settings_add_field(
                'post_statuses', $section,
                __( 'Post statuses', 'rs-link-checker' ),
                'settings_field_select_html',
                [
                    'values' => $values,
                    'multiple' => true,
                    'size' => -1,
                ]
            );

            $this->settings_add_field(
                'post_batch_size', $section,
                __( 'Post batch size', 'rs-link-checker' ),
                'settings_field_select_html',
                [
                    'values' => [
                        10 => '10',
                        20 => '20',
                        50 => '50',
                        100 => '100',
                    ],
                ]
            );


            // Links section

            $section = $this->settings_add_section(
                'section_links',
                __( 'Link related settings', 'rs-link-checker' ),
                __( 'Links can go stale at any time, and will need to be rechecked every once in a while. Select how often you want links rechecked, and in how large batches.', 'rs-link-checker' )
            );

            $this->settings_add_field(
                'link_interval', $section,
                __( 'Link re-check period', 'rs-link-checker' ),
                'settings_field_select_html',
                [
                    'values' => [
                        DAY_IN_SECONDS => __( 'Every day', 'rs-link-checker' ),
                        3 * DAY_IN_SECONDS => __( 'Every third day', 'rs-link-checker' ),
                        WEEK_IN_SECONDS => __( 'Every week', 'rs-link-checker' ),
                        2 * WEEK_IN_SECONDS => __( 'Every two weeks', 'rs-link-checker' ),
                    ],
                ]
            );


            $this->settings_add_field(
                'link_batch_size', $section,
                __( 'Link batch size', 'rs-link-checker' ),
                'settings_field_select_html',
                [
                    'default' => 20,
                    'values' => [
                        10 => '10',
                        20 => '20',
                        50 => '50',
                        100 => '100',
                    ],
                ]
            );

            $section = $this->settings_add_section(
                'section_blacklist',
                __( 'Blacklist settings', 'rs-link-checker' ),
                __( 'Blacklist links that should not be check. Each line is a pattern that can match anywhere in the URL of the link.', 'rs-link-checker' )
            );

            $this->settings_add_field(
                'link_blacklist_text', $section,
                __( 'Link blacklist', 'rs-link-checker' ),
                'settings_field_textarea_html',

            );

            $this->settings_add_field(
                'timeout', $section,
                __( 'Timeout for link requests', 'rs-link-checker' ),
                'settings_field_select_html',
                [
                    'default' => '5',
                    'values' => [
                        '5' => '5 seconds',
                        '10' => '10 seconds',
                        '15' => '15 seconds',
                        '30' => '30 seconds',
                    ],
                ]
            );

            $field = 'user_agent';
            $this->settings_add_field(
                $field, $section,
                __( 'User-Agent', 'rs-link-checker' ),
                'settings_field_input_html',
                [
                    'default' => $this->config[$field],
                ]
            );


            $this->settings_add_field(
                'logging', $section,
                __( 'Logging', 'rs-link-checker' ),
                'settings_field_checkbox_html'
            );
        }
    }



    /************************************************************************
     *
     *	Link tables
     *
     ************************************************************************/

    class Link_Checker_Table extends \WP_List_Table {
        protected $plugin = NULL;
        protected $page;
        protected $filter;

        protected $link_map;    // Map term_id to link
        protected $table_data;  // Display data

        function __construct( $plugin, $page, $filter = 'broken' ) {
            parent::__construct();

            $this->plugin = $plugin;
            $this->page = $page;

            $this->link_map = [];
            $this->table_data = [];
        }

        private function build_admin_url( $page, $args ) {
            return add_query_arg( $args, admin_url( $page ) );
        }

        private function req_var_raw( $name, $default = NULL ) {
            return ( $_REQUEST[$name] ?? $default );
        }

        private function req_var( $name, $default = NULL ) {
            return sanitize_key( $this->req_var_raw( $name, $default) );
        }

        function get_columns() {
            $columns = [
                'cb' => '<input type="checkbox" />',
                'url' => __('URL', 'backlinks-taxonomy'),
                'status' => __('Status', 'backlinks-taxonomy'),
                'message' => __('Message', 'backlinks-taxonomy'),
                'date' => __('Next check', 'backlinks-taxonomy'),
                'used' => __('Used in', 'backlinks-taxonomy'),
                'redirect' => __('Redirect', 'backlinks-taxonomy'),
            ];

            return $columns;
        }

        function set_links( $links ) {
        }

        function get_views() {
            $counts = $this->plugin->link_group_counts();
            $labels = $this->plugin->link_group_labels();

            $links = [];
            foreach ( $labels as $key => $label ) {
                if ( 0 == $counts[ $key ] )
                    continue;

                $data = [
                    'url' => $this->plugin->management_page_link( $key, true ),
                    'label' => sprintf( '%s (%d)', $label, $counts[ $key ] ),
                ];

                if ( $key == $this->filter )
                    $data['current'] = true;

                $links[] = $data;
            }

            return $this->get_views_links( $links );
        }

        function output_hidden_fields() {
            if ( $this->filter )
                printf( '<input type="hidden" name="filter" value="%s" />', esc_attr( $this->filter ) );
        }

        // Bind table with columns, data and all
        function prepare_items() {
            // Get table data
            $this->filter = $this->req_var( 'filter', 'broken' );
            $links = $this->plugin->get_links_filtered( $this->filter );

            $s = $this->req_var_raw( 's' );
            if ( $s ) {
                $search = strtolower( $s );
                $func = fn( $link ) => str_contains( strtolower( get_term_meta( $link->term_id, 'url', true ) ), $search );
                $links = array_filter( $links, $func );
            }

            $this->table_data = [];
            $this->link_map = [];
            foreach ( $links as $link ) {
                $this->link_map[ $link->term_id ] = $link;

                $meta = $this->plugin->get_term_meta( $link );

                $this->table_data[] = [
                    'id' => $link->term_id,
                    'url' => $meta[ 'url' ],
                    'status' => $meta[ 'status' ] ?? __( 'Unchecked', 'rs-link-checker' ),
                    'message' => $meta[ 'message' ] ?? '',
                    'redirect' => $meta[ 'redirect' ] ?? '',
                    'date' => $meta[ 'next_check'] ?? '',
                    'used' => $this->plugin->get_link_posts( $link ),
                ];
            }

            $this->sort_table_rows();

            // Do the columns and stuff
            $columns = $this->get_columns();
            $usermeta = get_user_meta( get_current_user_id(), "manage{$this->page}columnshidden", true);
            $hidden = ( is_array( $usermeta ) ? $usermeta : [] );
            $sortable = $this->get_sortable_columns();
            $primary  = 'title';

            $redirects = array_filter( $this->table_data, fn($item) => ! empty( $item['redirect'] ) );
            if ( empty( $redirects ) )
                $hidden[] = 'redirect';

            $this->_column_headers = [ $columns, $hidden, $sortable, $primary ];


            // Pagination
            $per_page = $this->get_items_per_page( 'elements_per_page', 10);
            $current_page = $this->get_pagenum();
            $total_items = count($this->table_data);

            $this->table_data = array_slice( $this->table_data, ( ($current_page - 1) * $per_page ), $per_page );

            $this->set_pagination_args( [
                'total_items' => $total_items, // total number of items
                'per_page'    => $per_page, // items to show on a page
                'total_pages' => ceil( $total_items / $per_page ) // use ceil to round up
            ] );

            $this->items = $this->table_data;
        }

        // Default set value for each column
        function column_default( $item, $column_name )  {
            return $item[$column_name] ?? '';
        }

        // Add a checkbox in the first column
        function column_cb( $item )  {
            return sprintf( '<input type="checkbox" name="ids[]" value="%d" />',  $item['id'] );
        }

        // Adding action links to title column
        function column_url( $item ) {
            $actions = [];

            $nonce = wp_create_nonce( 'rslc-row-actions-nonce' );
            $return = urlencode( $this->plugin->management_page_link( NULL, true ) );

            $bulk_actions = $this->plugin->get_row_actions();

            foreach ( $bulk_actions as $action => $data ) {
                $ok = call_user_func( $data['check'], $this->link_map[ $item['id'] ] );

                if ( $ok ) {
                    $url = $this->build_admin_url( 'admin-post.php', [
                        'action' => "rslc_$action",
                        'link' => $item['id'],
                        '_wpnonce' => $nonce,
                        'return' => $return,
                    ] );

                    $class = "rs-link-checker-$action";
                    $actions[$action] = sprintf( '<a href="%s" class="%s">%s</a>',
                                                 esc_url( $url ),
                                                 esc_attr( $class ),
                                                 esc_html( $data['label'] ) );
                }
            }

            return sprintf( '<a target=_blank href="%s" class="rslc-list-url">%s</a> %s',
                            esc_url( $item['url'] ), esc_html( $item['url'] ),
                            $this->row_actions( $actions )
            );
        }

        function column_date( $item ) {
            $timestamp = (int)$item['date'];
            return $timestamp
                ? date( 'Y-m-d H:i', $timestamp )
                : __( 'Never', 'rs-link-checker' )
                ;
        }

        function column_message( $item ) {
            $message = $item[ 'message' ];
            $message = preg_replace( '/^cURL error \d+: /', '', $message );
            return $message;
        }

        function column_used( $item ) {
            $post_types = [];
            foreach ( $this->plugin->get_post_types() as $post_type ) {
                $posts = array_filter(  $item['used'], fn( $post ) => $post->post_type == $post_type );
                if ( $posts )
                    $post_types[ $post_type ] = count( $posts );
            }

            $output = [];
            foreach ( $post_types as $post_type => $count ) {
                $pto = get_post_type_object( $post_type );
                $labels = $pto->labels;

                $term = get_term( $item['id'], $this->plugin->get_taxonomy_name() );
                $url = $this->build_admin_url( 'edit.php', [
                    'post_type' => $post_type,
                    'link_checker' => $term->name,
                ]  );

                $label = strtolower( ( $count == 1 ) ? $labels->singular_name : $labels->name );


                $output[] = sprintf( '<a href="%s">%d %s</a>', esc_url( $url), $count, esc_html( $label ) );
            }

            return join( '<br/>', $output );
        }


        protected function get_sortable_columns() {
            $sortable_columns = [
                'url'  => [ 'status', false ],
                'status'  => [ 'status', false ],
                'date'  => [ 'date', true ],
                'message'  => [ 'message', false ],
            ];
            return $sortable_columns;
        }

        protected function is_numeric_column( $column ) {
            $numeric_columns = [
                'status' => true,
            ];
            return $numeric_columns[ $column ] ?? false;
        }

        protected function sort_table_rows() {

            $orderby = ( !empty( $_GET['orderby'] ) ? sanitize_key( $_GET['orderby'] ) : 'url' );


            $order = ( !empty( $_GET['order'] ) ? sanitize_key( $_GET['order'] ) : 'asc' );

            if ( $this->is_numeric_column( $orderby ) )
                if ( $order == 'asc' )
                    $cmp = fn($a, $b) => $a[$orderby] <=> $b[$orderby];
                else
                    $cmp = fn($a, $b) => $b[$orderby] <=> $a[$orderby];
            else
                if ( $order == 'asc' )
                    $cmp = fn($a, $b) => strcmp( $a[$orderby], $b[$orderby] );
                else
                    $cmp = fn($a, $b) => strcmp( $b[$orderby], $a[$orderby] );

            usort( $this->table_data, $cmp );
        }



        function get_bulk_actions() {
            $bulk_actions = $this->plugin->get_row_actions();

            $actions = [];
            foreach ( $bulk_actions as $action => $data )
                $actions[$action] = $data['label'];

            return $actions;
        }

        function do_bulk_actions() {
            $action = $this->current_action();
            if ( $action )
                do_action( "link_checker_page_bulk_{$action}" );
        }
    }

    global $RSLinkCheckerPlugin;
    $RSLinkCheckerPlugin = new LinkCheckerPlugin();

    if ( defined( 'WP_CLI' ) && WP_CLI ) {
        require_once dirname( __FILE__ ) . '/rs-link-checker-cli.php';
    }

} );
