<?php
/**
 * Plugin Name:         WordPress Plugin Site
 * Description:         A helper plugin for https://plugins.seindal.dk/
 * Plugin URI:          https://plugins.seindal.dk/plugins/wordpress-plugin-site/
 * Update URI:          https://plugins.seindal.dk
 * Author:              René Seindal
 * Author URI:          https://plugins.seindal.dk/
 * Donate link:         https://mypos.com/@historywalks
 * Requires Plugins:    rs-base-plugin
 * License:             GPL v2 or later
 * License URI:         https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:         wordpress-plugin-site
 * Domain Path:         /languages
 * Requires PHP:        7.4
 * Requires at least:   5.0
 * Version:             1.0.9
 **/

namespace ReneSeindal;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly


add_action( 'plugins_loaded', function() {
    class PluginSite extends PluginBase {
        protected $plugin_file = __FILE__;

        protected $download_path = 'download';
        protected $download_dir = ABSPATH . 'download';

        // Media sizes for icons and banners
        protected $media_sizes = [
            [ 'icons', '1x', 128, 128 ],
            [ 'icons', '2x', 256, 256 ],
            [ 'banners', 'low', 772, 250 ],
            [ 'banners', 'high', 1544, 500 ],
        ];

        use PluginBaseSettings;

        protected $defaults = [
            "Donate link" => "https://mypos.com/@historywalks",
            "License URI" => "https://www.gnu.org/licenses/gpl-2.0.html",
            "License" => "GPL v2 or later",
            "Requires PHP" => "7.4",
            "Requires at least" => "5.0",
        ];

        /************************************************************
         *
         * Fix searches on taxonomy archives to include all the post
         * types associated with the taxonomy
         *
         ************************************************************/

        function do_pre_get_posts_action_taxonomy_archives( $wp_query ) {
            if ( $wp_query->is_main_query() && ! is_admin() ) {
                $queried_object = $wp_query->get_queried_object();
                if ( is_a( $queried_object, 'WP_Term' ) ) {
                    $taxonomy = get_taxonomy( $queried_object->taxonomy );
                    $wp_query->set( 'post_type', $taxonomy->object_type );
                }
            }
        }

        /************************************************************
         *
         * Lookup page by plugin slug
         *
         ************************************************************/

        function get_plugin_post( $slug ) {
            $query = [
                'post_status' => 'publish',
                'post_type' => 'rsplugin',
                'numberposts' => 1,
                'name' => $slug,
            ];

            $posts = get_posts( $query );
            return empty( $posts ) ? NULL : $posts[0];
        }


        /************************************************************
         *
         * Find banner and icon images
         *
         ************************************************************/

        function get_media_by_parent( $parent ) {
            $query = [
                'post_type' => 'attachment',
                'post_status' => 'inherit',
                'post_mime_type' => [ 'image/png',  'image/jpeg', 'image/gif', 'image/avif' ],
                'post_parent' => $parent,
            ];

            return get_posts( $query );
        }

        function get_plugin_media( $post ) {
            $post = get_post( $post );

            // Parents in falling priority - post,home,none
            $parents = [ $post->ID ];
            $front = absint( get_option( 'page_on_front' ) );
            if ( $front ) $parents[1] = $front;
            $parents[] = 0;

            $output = [ 'icons' => [], 'banners' => [] ];

            foreach ( $parents as $parent ) {
                $media = $this->get_media_by_parent( $parent );

                foreach ( $media as $medium ) {
                    $meta = wp_get_attachment_metadata( $medium->ID );
                    if ( false !== $meta ) {
                        foreach ( $this->media_sizes as $size ) {
                            if ( empty( $output[ $size[0] ][ $size[1] ] ) ) {
                                if ( $meta['width'] == $size[2] && $meta['height'] == $size[3] ) {
                                    $output[ $size[0] ][ $size[1] ] = wp_get_shortlink( $medium );
                                }
                            }
                        }
                    }
                }
            }

            return $output;
        }

        /************************************************************
         *
         * Generate readme.txt snippets
         *
         ************************************************************/

        function do_init_action_rewrite_endpoint_readme() {
            add_rewrite_endpoint( 'readme', EP_PAGES );
        }

        function do_template_redirect_action_readme() {
            global $wp_query;

            // if this is not a request for json or a singular object then bail
            if ( ! isset( $wp_query->query_vars['readme'] ) || ! is_singular() )
                return;

            $post = get_queried_object();

            if ( ! is_a( $post, 'WP_Post' ) )
                wp_die ( __( 'This shouldn\'t happen!' ) );

            header('Content-Type: text/plain');
            $this->generate_readme_snippet( $post, $wp_query->query_vars['readme'] );
            exit;
        }

        function generate_readme_snippet( $post, $stable_tag = NULL ) {
            $post = get_post( $post );
            if ( ! $post ) return;

            $meta = $this->generate_meta_data( $post, $_GET );

            if ( $stable_tag )
                $meta['Stable tag'] = $stable_tag;

            $fields = [
                "Contributors",
                "Tags",
                "Requires at least",
                "Tested up to",
                "Requires PHP",
                "Stable tag",
                "License",
                "License URI",
                "Donate link",
            ];

            printf( "=== %s ===\n", $meta['Plugin Name'] );
            foreach ( $fields as $field )
                if ( !empty( $meta[$field] ) )
                    printf( "%s: %s\n", $field, $meta[$field] );
            printf( "\n" );

            $error_mode = libxml_use_internal_errors( true );

            $xml = new \DOMDocument( '1.0', 'UTF-8' );

            if ( $xml->LoadHTML( $post->post_content, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD ) ) {
                foreach ( $xml->childNodes as $node ) {
                    // error_log( "nodeName {$node->nodeName}" );
                    switch ( $node->nodeName ) {
                    case 'p' :
                        foreach ( $node->childNodes as $child ) {
                            switch ( $child->nodeName ) {
                            case 'b':
                            case 'strong':
                                echo '**', $child->textContent, '**';
                                break;
                            case 'i':
                            case 'em':
                                echo '*', $child->textContent, '*';
                                break;
                            case 'code':
                                echo '`', $child->textContent, '`';
                                break;
                            case 'a':
                                $href = $child->getAttribute( 'href' );
                                if ( $href )
                                    printf( '[%s](%s)', $child->textContent, $href );
                                else
                                    echo $child->textContent;
                                break;
                            default:
                                echo $child->textContent;
                            }
                        }
                        printf( "\n\n" );
                        break;
                    case 'pre':
                        echo "\t", str_replace( "\n", "\n\t", $node->textContent ), "\n\n";
                        break;
                    case 'h2':
                        printf( "== %s ==\n\n", $node->textContent );
                        break;
                    case 'h3':
                        printf( "= %s =\n\n", $node->textContent );
                        break;
                    case 'ul':
                        foreach ( $node->childNodes as $item ) {
                            if ( 'li' == $item->nodeName ) {
                                printf( "* %s\n", $item->textContent );
                            }
                        }
                        print "\n";
                        break;
                    case 'ol':
                        $number = 1;
                        foreach ( $node->childNodes as $item ) {
                            if ( 'li' == $item->nodeName ) {
                                printf( "%d. %s\n", $number++, $item->textContent );
                            }
                        }
                        print "\n";
                        break;
                    default:
                        break;
                    }

                    // Stop at a "Read more" marker
                    if ( '#comment' == $node->nodeName and 'more' == $node->textContent )
                        break;
                }
            }

            libxml_use_internal_errors( $error_mode );
        }

        /************************************************************
         *
         * Generate meta data for plugin
         *
         ************************************************************/

        function do_init_action_rewrite_endpoint_meta() {
            add_rewrite_endpoint( 'meta', EP_PAGES );
        }

        function do_template_redirect_action_meta() {
            global $wp_query;

            // if this is not a request for json or a singular object then bail
            if ( ! isset( $wp_query->query_vars['meta'] ) || ! is_singular() )
                return;

            $post = get_queried_object();

            if ( ! is_a( $post, 'WP_Post' ) )
                wp_die ( __( 'This shouldn\'t happen!' ) );

            $meta = $this->generate_meta_data( $post, $_GET );

            header('Content-Type: application/json');
            echo json_encode( $meta, JSON_PRETTY_PRINT );
            exit;
        }


        function generate_meta_data( $post, $override = [] ) {
            $user = get_userdata( $post->post_author );
            $tags = get_the_tags( $post );

            $meta = [
                "Author URI" => $user->data->user_url ?: site_url(),
                "Author" => $user->data->display_name,
                "Contributors" => $user->data->user_login,
                "Description" => $post->post_excerpt,
                "Plugin Name" => $post->post_title,
                "Plugin URI" => get_permalink( $post ),
                "Text Domain" => $post->post_name,
                "Domain Path" => "/languages",
                "Update URI" => site_url(),
                "Tags" => join( ', ', wp_list_pluck( $tags, 'name' ) ),
                "Tested up to" => $this->wp_version_tested(),
                "Stable tag" => NULL,
                "Version" => NULL,
            ];

            foreach ( array_keys( $this->defaults ) as $key ) {
                $value = $this->get_option( $key );
                if ( !empty( $value ) && $this->validate_setting( $key, $value ) !== false )
                    $meta[$key] = $value;
                else
                    $meta[$key] = $this->defaults[$key];
            }

            $required = get_the_terms( $post, 'plugin-requirements' );
            if ( $required and !is_wp_error( $required ) )
                $meta['Requires Plugins'] = join( ', ', array_map( fn($term) => $term->slug, $required ) );

            foreach ( get_the_terms( $post, 'category' ) as $cat )
                if ( $cat->slug === 'wordpress-plugin-registry' )
                    unset( $meta['Update URI'] );

            foreach ( $override as $key => $value ) {
                $key = sanitize_text_field( $key );

                if ( array_key_exists( $key, $meta ) )
                    $meta[$key] = sanitize_text_field( $value );
                else {
                    $newkey = strtr( $key, '_', ' ' );
                    if ( array_key_exists( $newkey, $meta ) )
                        $meta[$newkey] = sanitize_text_field( $value );
                }
            }

            return $meta;
        }


        /************************************************************
         *
         * Plugin API info
         *
         ************************************************************/

        function do_init_action_rewrite_endpoint_info() {
            add_rewrite_endpoint( 'info', EP_PAGES );
        }

        function do_template_redirect_action_info() {
            global $wp_query;

            // if this is not a request for json or a singular object then bail
            if ( ! isset( $wp_query->query_vars['info'] ) || ! is_singular() )
                return;

            $post = get_queried_object();

            if ( ! is_a( $post, 'WP_Post' ) )
                wp_die ( __( 'This shouldn\'t happen!' ) );

            $info = $this->generate_plugin_info( $post, $wp_query->query_vars['info'] );

            header('Content-Type: application/json');
            echo json_encode( $info, JSON_PRETTY_PRINT );
            exit;
        }


        function generate_plugin_info( $post, $version ) {
            $slug = $post->post_name;
            $user = get_userdata( $post->post_author );

            $downloads = $this->get_plugin_download_list();
            if ( empty( $downloads[$slug] ) )
                return NULL;

            $download = $downloads[$slug]['stable'] ?? $downloads[$slug]['latest'];

            $content = ( explode( '<!--more-->', apply_filters( 'the_content', $post->post_content ) ) )[0];
            $regex = '/(<h2 class="wp-block-heading">.*<\/h2>)/i';
            $parts = preg_split( $regex, $content, -1, PREG_SPLIT_DELIM_CAPTURE );

            $sections = [];
            while ( $part = array_shift( $parts ) ) {
                if ( str_contains( $part, 'wp-block-heading' ) ) {
                    $body = array_shift( $parts );
                    if ( !empty( trim($body) ) ) {
                        $part = strip_tags($part);
                        $sections[ sanitize_title( $part ) ] = sprintf( '<h3>%s</h3>', $part) .  $body;
                    }
                }
            }

            if ( empty( $sections ) )
                $sections['description'] = $post->post_excerpt;

            $media = $this->get_plugin_media( $post );

            return [
                'name' => $post->post_title,
                'slug' => $slug,
                'version' => $download['version'],
                'date' => $download['date'],
                'author' => sprintf( '<a href="%s">%s</a>', esc_url( $user->user_url ), esc_html( $user->display_name ) ),
                'requires' => '6.0',
                'tested' => $this->wp_version_tested(),
                'homepage' => get_permalink( $post ),
                'downloaded' => '1',
                'external' => '',
                'package' => $download['url'],
                'file_name' => $download['file'],
                'short_description' => $post->post_excerpt,
                'sections' => $sections,
                'icons' => $media['icons'],
                'banners' => $media['banners'],
                'banners_rtl' => [],
                'ratings' => [ '5' => 1, '4' => 0, '3' => 0, '2' => 0, '1' => 0 ],
                'num_ratings' => 1,

                'rating' => 5,
                'last_updated' => $download['date'],
                'active_installs' => 1,

                'download_link' => $download['package'],
            ];
        }

        /************************************************************
         *
         * Download lastest version shortcode
         *
         ************************************************************/

        function do_download_link_shortcode( $atts = [], $content = null, $tag = '' ) {
            // normalize attribute keys, lowercase
            $atts = array_change_key_case( (array)$atts, CASE_LOWER );

            // override default attributes with user attributes
            $atts = shortcode_atts( [
                'plugin' => NULL,
                'version' => NULL,
            ] , $atts, $tag );

            $post = get_post();

            $plugin = $atts['plugin'] ?? $post->post_name;

            $downloads = $this->get_plugin_download_list();

            if ( empty( $downloads[$plugin] ) )
                return __( 'No versions available for download.', 'wordpress-plugin-site' );

            $version = $atts['version'] ?? 'stable';
            $version_number = NULL;

            if ( $version == 'stable' ) {
                if ( empty( $downloads[$plugin][$version] ) )
                    return __( 'No stable version available for download.', 'wordpress-plugin-site' );
                $version_number = $downloads[$plugin][$version]['version'];
                $download_text = __( 'Download stable version %s', 'wordpress-plugin-site' );
            }
            elseif ( $version == 'latest' ) {
                if ( empty( $downloads[$plugin][$version] ) )
                    return __( 'No lastest version available for download.', 'wordpress-plugin-site' );
                $version_number = $downloads[$plugin][$version]['version'];
                $download_text = __( 'Download lastest version %s', 'wordpress-plugin-site' );
            }
            else {
                if ( empty( $downloads[$plugin][$version] ) )
                    return __( 'No such version available for download.', 'wordpress-plugin-site' );
                $version_number = $downloads[$plugin][$version]['version'];
                $download_text = __( 'Download version %s', 'wordpress-plugin-site' );
            }

            if ( empty( $content ) )
                $content = sprintf( $download_text, $version_number );

            return sprintf( '<a href="%s">%s</a>.',
                            esc_url( $downloads[$plugin][$version]['package'] ),
                            esc_html( $content )
            );

        }


        /************************************************************
         *
         * Get list of plugins and versions in the download folder
         *
         ************************************************************/

        function get_plugin_download_list() {
            $plugins = [];
            $sortkeys = [];
            foreach ( scandir( $this->download_dir ) as $file ) {
                if ( preg_match('/^(.*)-((?:\d+)(?:\.\d+)+)\.zip$/', $file, $m ) ) {
                    $slug = $m[1];
                    $version = $m[2];

                    $post = $this->get_plugin_post( $slug );
                    if ( !$post ) continue;

                    $media = $this->get_plugin_media( $post );

                    $data = [
                        'slug'			=> $slug,
                        'plugin'		=> "$slug/$slug.php",
                        'new_version'	=> $version,
                        'url'           => get_permalink( $post ),
                        'package'		=> site_url( "/{$this->download_path}/$file" ),
                        'icons'			=> $media['icons'],
                        'banners'		=> $media['banners'],
                        'banners_rtl'   => [],
                        'tested'		=> $this->wp_version_tested(),
                        'requires_php'  => '8.0',

                        'file' => $file,
                        'date' => date( 'Y-m-d G:i:s', filemtime( "{$this->download_dir}/$file" ) ),
                        'version'	=> $version,

                        // Not WP standard
                        'name' => $post->post_title,
                        'short_description' => $post->post_excerpt,
                    ];

                    $plugins[ $m[1] ][ $version ] = $data;
                }
            }

            foreach ( $plugins as $plugin => &$versions ) {
                usort( $versions, fn($a,$b) => version_compare( $a['version'], $b['version'] ) );

                $latest = $versions[ count($versions) - 1 ];
                $zip = $latest['file'];
                $plugin = $latest['slug'];
                $readme = @file( "zip://{$this->download_dir}/$zip#$plugin/readme.txt");

                if ( $readme ) {
                    $stable = NULL;
                    foreach ( $readme as $line ) {
                        if ( preg_match( '/stable tag:\s*([\d.]+)/i', $line, $m ) )
                            $stable = $m[1];
                    }

                    if ( $stable ) {
                        $found = array_filter( $versions, fn( $version ) => version_compare( $version['version'], $stable, '=' ) );
                        if ( $found ) {
                            $versions['stable'] = ( array_values( $found ) )[0];

                        }
                    }
                }

                $versions['latest'] = $latest;

            }
            return $plugins;
        }



        /************************************************************
         *
         * Public endpoint for list of plugins and versions
         *
         ************************************************************/

        function do_init_action_rewrite_endpoint_plugin_index() {
            add_rewrite_endpoint( 'plugin-index', EP_ROOT );
        }

        function do_template_redirect_action_plugin_index() {
            global $wp_query;

            if ( ! isset( $wp_query->query_vars['plugin-index'] ) )
                return;

            header('Content-Type: application/json');

            $downloads = $this->get_plugin_download_list();
            echo json_encode( $downloads, JSON_PRETTY_PRINT );

            exit;
        }


        function wp_version_tested() {
            global $wp_version;

            // In case of beta or RC versions
            $version = (explode( '-', $wp_version ))[0];

            // Return two levels
            return join( '.', array_slice( explode( '.', $version ), 0, 2 ) );
        }


        /************************************************************
         *
         * Gutenberg patterns - remove all standard patterns
         *
         ************************************************************/

        function do_should_load_remote_block_patterns_filter() {
            return false;
        }

        function do_after_setup_theme_action_patterns() {
            remove_theme_support( 'core-block-patterns' );
        }

        protected $init_remove_core_block_patterns_callback_priority = 999;

        function Xdo_init_action_remove_core_block_patterns() {
            $registered_patterns = \WP_Block_Patterns_Registry::get_instance()->get_all_registered();
            if ( $registered_patterns ) {
                foreach ( $registered_patterns as $pattern_properties ) {
                    unregister_block_pattern( $pattern_properties['name'] );
                }
            }
        }



        /************************************************************
         *
         * Post type 'rsplugin' for plugin descriptions
         *
         ************************************************************/

        function do_init_action_register_rsplugin() {
            $supports = [
                'title',
                'editor' => [ 'notes' => true ],
                'comments',
                'revisions',
                'author',
                'excerpt',
                'page-attributes',
                'thumbnail',
                'custom-fields',
            ];

            register_post_type(
                'rsplugin',
                [
                    'labels'                => [
                        'name'                  => __( 'Plugins', 'wordpress-plugin-site' ),
                        'singular_name'         => __( 'Plugin', 'wordpress-plugin-site' ),
                        'all_items'             => __( 'All Plugins', 'wordpress-plugin-site' ),
                        'archives'              => __( 'Plugin Archives', 'wordpress-plugin-site' ),
                        'attributes'            => __( 'Plugin Attributes', 'wordpress-plugin-site' ),
                        'insert_into_item'      => __( 'Insert into Plugin', 'wordpress-plugin-site' ),
                        'uploaded_to_this_item' => __( 'Uploaded to this Plugin', 'wordpress-plugin-site' ),
                        'featured_image'        => _x( 'Featured Image', 'rsplugin', 'wordpress-plugin-site' ),
                        'set_featured_image'    => _x( 'Set featured image', 'rsplugin', 'wordpress-plugin-site' ),
                        'remove_featured_image' => _x( 'Remove featured image', 'rsplugin', 'wordpress-plugin-site' ),
                        'use_featured_image'    => _x( 'Use as featured image', 'rsplugin', 'wordpress-plugin-site' ),
                        'filter_items_list'     => __( 'Filter Plugins list', 'wordpress-plugin-site' ),
                        'items_list_navigation' => __( 'Plugins list navigation', 'wordpress-plugin-site' ),
                        'items_list'            => __( 'Plugins list', 'wordpress-plugin-site' ),
                        'new_item'              => __( 'New Plugin', 'wordpress-plugin-site' ),
                        'add_new'               => __( 'Add New', 'wordpress-plugin-site' ),
                        'add_new_item'          => __( 'Add New Plugin', 'wordpress-plugin-site' ),
                        'edit_item'             => __( 'Edit Plugin', 'wordpress-plugin-site' ),
                        'view_item'             => __( 'View Plugin', 'wordpress-plugin-site' ),
                        'view_items'            => __( 'View Plugins', 'wordpress-plugin-site' ),
                        'search_items'          => __( 'Search Plugins', 'wordpress-plugin-site' ),
                        'not_found'             => __( 'No Plugins found', 'wordpress-plugin-site' ),
                        'not_found_in_trash'    => __( 'No Plugins found in trash', 'wordpress-plugin-site' ),
                        'parent_item_colon'     => __( 'Parent Plugin:', 'wordpress-plugin-site' ),
                        'menu_name'             => __( 'Plugins', 'wordpress-plugin-site' ),
                    ],
                    'public'                => true,
                    'hierarchical'          => false,
                    'show_ui'               => true,
                    'show_in_nav_menus'     => true,
                    'supports'              => $supports,
                    'has_archive'           => false,
                    'rewrite'               => [ 'slug' => 'plugins', 'ep_mask' => EP_PAGES ],
                    'query_var'             => true,
                    'menu_position'         => null,
                    'menu_icon'             => 'dashicons-plugins-checked',
                    'show_in_rest'          => true,
                    'rest_base'             => 'rsplugin',
                    'rest_controller_class' => 'WP_REST_Posts_Controller',
                ]
            );

            register_taxonomy_for_object_type( 'post_tag', 'rsplugin' );
            register_taxonomy_for_object_type( 'category', 'rsplugin' );
        }

        function do_post_updated_messages_filter_rsplugin( $messages ) {
            global $post;

            $permalink = get_permalink( $post );

            $messages['rsplugin'] = [
                0  => '', // Unused. Messages start at index 1.
                /* translators: %s: post permalink */
                1  => sprintf( __( 'Plugin updated. <a target="_blank" href="%s">View Plugin</a>', 'wordpress-plugin-site' ), esc_url( $permalink ) ),
                2  => __( 'Custom field updated.', 'wordpress-plugin-site' ),
                3  => __( 'Custom field deleted.', 'wordpress-plugin-site' ),
                4  => __( 'Plugin updated.', 'wordpress-plugin-site' ),
                /* translators: %s: date and time of the revision */
                5  => isset( $_GET['revision'] ) ? sprintf( __( 'Plugin restored to revision from %s', 'wordpress-plugin-site' ), wp_post_revision_title( (int) $_GET['revision'], false ) ) : false, // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                /* translators: %s: post permalink */
                6  => sprintf( __( 'Plugin published. <a href="%s">View Plugin</a>', 'wordpress-plugin-site' ), esc_url( $permalink ) ),
                7  => __( 'Plugin saved.', 'wordpress-plugin-site' ),
                /* translators: %s: post permalink */
                8  => sprintf( __( 'Plugin submitted. <a target="_blank" href="%s">Preview Plugin</a>', 'wordpress-plugin-site' ), esc_url( add_query_arg( 'preview', 'true', $permalink ) ) ),
                /* translators: 1: Publish box date format, see https://secure.php.net/date 2: Post permalink */
                9  => sprintf( __( 'Plugin scheduled for: <strong>%1$s</strong>. <a target="_blank" href="%2$s">Preview Plugin</a>', 'wordpress-plugin-site' ), date_i18n( __( 'M j, Y @ G:i', 'wordpress-plugin-site' ), strtotime( $post->post_date ) ), esc_url( $permalink ) ),
                /* translators: %s: post permalink */
                10 => sprintf( __( 'Plugin draft updated. <a target="_blank" href="%s">Preview Plugin</a>', 'wordpress-plugin-site' ), esc_url( add_query_arg( 'preview', 'true', $permalink ) ) ),
            ];

            return $messages;
        }


        // Sets the bulk post updated messages for the `rsplugin` post type.

        protected $bulk_post_updated_messages_callback_arguments = 2;

        function do_bulk_post_updated_messages_filter_rsplugin( $bulk_messages, $bulk_counts ) {
            global $post;

            $bulk_messages['rsplugin'] = [
                /* translators: %s: Number of Plugins. */
                'updated'   => _n( '%s Plugin updated.', '%s Plugins updated.', $bulk_counts['updated'], 'wordpress-plugin-site' ),
                'locked'    => ( 1 === $bulk_counts['locked'] ) ? __( '1 Plugin not updated, somebody is editing it.', 'wordpress-plugin-site' ) :
                /* translators: %s: Number of Plugins. */
                _n( '%s Plugin not updated, somebody is editing it.', '%s Plugins not updated, somebody is editing them.', $bulk_counts['locked'], 'wordpress-plugin-site' ),
                /* translators: %s: Number of Plugins. */
                'deleted'   => _n( '%s Plugin permanently deleted.', '%s Plugins permanently deleted.', $bulk_counts['deleted'], 'wordpress-plugin-site' ),
                /* translators: %s: Number of Plugins. */
                'trashed'   => _n( '%s Plugin moved to the Trash.', '%s Plugins moved to the Trash.', $bulk_counts['trashed'], 'wordpress-plugin-site' ),
                /* translators: %s: Number of Plugins. */
                'untrashed' => _n( '%s Plugin restored from the Trash.', '%s Plugins restored from the Trash.', $bulk_counts['untrashed'], 'wordpress-plugin-site' ),
            ];

            return $bulk_messages;
        }


        /************************************************************
         *
         * Taxonomy 'plugin-requirements'
         *
         ************************************************************/

        function do_init_action_plugin_requirements() {
            register_taxonomy( 'plugin-requirements', [ 'rsplugin' ], [
                'hierarchical'          => false,
                'public'                => true,
                'show_in_nav_menus'     => true,
                'show_ui'               => true,
                'show_admin_column'     => true,
                'query_var'             => true,
                'rewrite'               => true,
                'capabilities'          => [
                    'manage_terms' => 'edit_posts',
                    'edit_terms'   => 'edit_posts',
                    'delete_terms' => 'edit_posts',
                    'assign_terms' => 'edit_posts',
                ],
                'labels'                => [
                    'name'                       => __( 'Requirements', 'wordpress-plugin-site' ),
                    'singular_name'              => _x( 'Requirements', 'taxonomy general name', 'wordpress-plugin-site' ),
                    'search_items'               => __( 'Search Requirements', 'wordpress-plugin-site' ),
                    'popular_items'              => __( 'Popular Requirements', 'wordpress-plugin-site' ),
                    'all_items'                  => __( 'All Requirements', 'wordpress-plugin-site' ),
                    'parent_item'                => __( 'Parent Requirements', 'wordpress-plugin-site' ),
                    'parent_item_colon'          => __( 'Parent Requirements:', 'wordpress-plugin-site' ),
                    'edit_item'                  => __( 'Edit Requirements', 'wordpress-plugin-site' ),
                    'update_item'                => __( 'Update Requirements', 'wordpress-plugin-site' ),
                    'view_item'                  => __( 'View Requirements', 'wordpress-plugin-site' ),
                    'add_new_item'               => __( 'Add New Requirements', 'wordpress-plugin-site' ),
                    'new_item_name'              => __( 'New Requirements', 'wordpress-plugin-site' ),
                    'separate_items_with_commas' => __( 'Separate Requirements with commas', 'wordpress-plugin-site' ),
                    'add_or_remove_items'        => __( 'Add or remove Requirements', 'wordpress-plugin-site' ),
                    'choose_from_most_used'      => __( 'Choose from the most used Requirements', 'wordpress-plugin-site' ),
                    'not_found'                  => __( 'No Requirements found.', 'wordpress-plugin-site' ),
                    'no_terms'                   => __( 'No Requirements', 'wordpress-plugin-site' ),
                    'menu_name'                  => __( 'Requirements', 'wordpress-plugin-site' ),
                    'items_list_navigation'      => __( 'Requirements list navigation', 'wordpress-plugin-site' ),
                    'items_list'                 => __( 'Requirements list', 'wordpress-plugin-site' ),
                    'most_used'                  => _x( 'Most Used', 'plugin-requirements', 'wordpress-plugin-site' ),
                    'back_to_items'              => __( '&larr; Back to Requirements', 'wordpress-plugin-site' ),
                ],
                'show_in_rest'          => true,
                'rest_base'             => 'plugin-requirements',
                'rest_controller_class' => 'WP_REST_Terms_Controller',
            ] );

        }

        // Sets the post updated messages for the `plugin_requirements` taxonomy.

        function do_term_updated_messages_filter_plugin_requirements( $messages ) {
            $messages['plugin-requirements'] = [
                0 => '', // Unused. Messages start at index 1.
                1 => __( 'Requirements added.', 'wordpress-plugin-site' ),
                2 => __( 'Requirements deleted.', 'wordpress-plugin-site' ),
                3 => __( 'Requirements updated.', 'wordpress-plugin-site' ),
                4 => __( 'Requirements not added.', 'wordpress-plugin-site' ),
                5 => __( 'Requirements not updated.', 'wordpress-plugin-site' ),
                6 => __( 'Requirements deleted.', 'wordpress-plugin-site' ),
            ];

            return $messages;
        }


        /************************************************************************
         *
         *	Settings
         *
         ************************************************************************/

        function do_admin_menu_action() {
            $this->settings_add_menu(
                __( 'Plugin Registry', 'wordpress-plugin-site' ),
                __( 'Plugin Registry', 'wordpress-plugin-site' )
            );
        }

        protected $wp_versions = [
            '5.0', '5.1', '5.2', '5.3', '5.4',
            '5.5', '5.6', '5.7', '5.8', '5.9',
            '6.0', '6.1', '6.2', '6.3', '6.4',
            '6.5'
        ];

        protected $php_versions = [
            '7.3',
            '7.4',
            '8.0',
            '8.1',
            '8.2',
            '8.3',
        ];

        function do_rswp_validate_donatelink_filter( $url ) {
            return wp_http_validate_url( $url ) ? $url : false;
        }

        function do_rswp_validate_licenseuri_filter( $url ) {
            return wp_http_validate_url( $url ) ? $url : false;
        }

        function do_rswp_validate_requiresphp_filter( $v  ) {
            return in_array( $v, $this->php_versions ) ? $v : false;
        }

        function do_rswp_validate_requiresatleast_filter( $v ) {
            return in_array( $v, $this->wp_versions ) ? $v : false;
        }

        function validate_setting( $key, $value = false ) {
            if ( $value === false)
                $value = $this->get_option( $key );
            if ( $value )
                $value = apply_filters( 'rswp_validate_' . sanitize_key( $key ), $value );
            return $value;
        }

        function settings_define_sections_and_fields() {
            $section = $this->settings_add_section(
                'section_metadata',
                __( 'Default meta-data', 'wordpress-plugin-site' ),
                __( 'Please specify the defaults for some of the meta-data fields for all plugins.', 'wordpress-plugin-site' )
            );

            $field = 'Donate link';
            $this->settings_add_field(
                $field, $section,
                __( 'Donate link', 'wordpress-plugin-site' ),
                'settings_field_input_html',
                [
                    'default' => $this->defaults[$field],
                ]
            );

            $field = 'License';
            $this->settings_add_field(
                $field, $section,
                __( 'License', 'wordpress-plugin-site' ),
                'settings_field_input_html',
                [
                    'default' => $this->defaults[$field],
                ]
            );

            $field = 'License URI';
            $this->settings_add_field(
                $field, $section,
                __( 'License URI', 'wordpress-plugin-site' ),
                'settings_field_input_html',
                [
                    'default' => $this->defaults[$field],
                ]
            );

            $section = $this->settings_add_section(
                'section_versions',
                __( 'Default WP and PHP versions', 'wordpress-plugin-site' ),
                __( 'Please specify the defaults for WP and PHP', 'wordpress-plugin-site' )
            );

            // Versions WP & PHP

            $field = 'Requires at least';
            $this->settings_add_field(
                $field, $section,
                __( 'Minimum WordPress version', 'wordpress-plugin-site' ),
                'settings_field_select_html',
                [
                    'default' => $this->defaults[$field],
                    'values' => array_combine( $this->wp_versions, $this->wp_versions ),
                ]
            );

            $field = 'Requires PHP';
            $this->settings_add_field(
                $field, $section,
                __( 'Minimum PHP version', 'wordpress-plugin-site' ),
                'settings_field_select_html',
                [
                    'default' => $this->defaults[$field],
                    'values' => array_combine( $this->php_versions, $this->php_versions ),
                ]
            );
        }

        function settings_sanitize_option( $settings = NULL ) {
            foreach ( $settings as $key => $value ) {
                if ( $this->validate_setting( $key, $value) === false )
                    add_settings_error(
                        $this->option_name, sanitize_key( $key ),
                        sprintf( __( '%s has an invalid value.', 'wordpress-plugin-site' ), esc_html( $key ) )
                    );
            }
            return $settings;
        }
    }

    PluginSite::instance();


    /************************************************************
     *
     * CLI interface
     *
     ************************************************************/

    if ( defined( 'WP_CLI' ) && WP_CLI ) {

        /**
         * Plugin Site CLI commands
         *
         * ## EXAMPLES
         *
         *     wp rswp readme backlinks-taxonomy
         *
         */

        class PluginSiteCLI {
            protected $plugin = NULL;

            function __construct( ) {
                $this->plugin = PluginSite::instance();
            }

            /**
             * Generate a readme.txt
             *
             * ## OPTIONS
             *
             * <plugin-slug>
             * : Generate readme.txt for this pluging
             *
             * ## EXAMPLES
             *
             *     wp rswp readme backlinks-taxonomy
             *
             */

            public function readme( $args, $assoc ) {
                $query = [
                    'post_status' => 'publish',
                    'post_type' => 'rsplugin',
                    'numberposts' => 1,
                    'name' => $args[0],
                ];

                $posts = get_posts( $query );
                if ( empty( $posts ) )
                    \WP_CLI::error( 'Post not found' );
                else
                    $this->plugin->generate_readme_snippet( $posts[0] );
            }

            /**
             * Generate meta data for a plugin
             *
             * ## OPTIONS
             *
             * <plugin-slug>
             * : Generate meta data for this pluging
             *
             * [<override>...]
             * : Override arguments in the format Key=Value
             *
             * ## EXAMPLES
             *
             *     wp rswp meta backlinks-taxonomy
             *
             */

            public function meta( $args, $assoc ) {
                $post = $this->plugin->get_plugin_post( array_shift( $args ) );

                if ( empty( $post ) )
                    \WP_CLI::error( 'Plugin not found' );
                else {
                    $override = [];
                    foreach ( $args as $arg ) {
                        list( $key, $value ) = explode( '=', $arg, 2 );
                        $override[$key] = $value;
                    }

                    echo json_encode( $this->plugin->generate_meta_data( $post, $override ), JSON_PRETTY_PRINT );
                }
            }

            /**
             * Get plugin download JSON
             *
             * ## EXAMPLES
             *
             *     wp rswp downloads
             *
             */

            public function downloads( $args, $assoc ) {
                $downloads = $this->plugin->get_plugin_download_list();

                if ( empty( $downloads ) )
                    \WP_CLI::error( 'Downloads not found' );
                else
                    echo json_encode( $downloads, JSON_PRETTY_PRINT );
            }

            /**
             * Find plugin icons and banners
             *
             * ## OPTIONS
             *
             * <plugin-slug>
             * : Find icons and banners for this plugin
             *
             * ## EXAMPLES
             *
             *     wp rswp media rs-scratchpad
             *
             */

            public function media( $args, $assoc ) {
                $query = [
                    'post_status' => 'publish',
                    'post_type' => 'rsplugin',
                    'numberposts' => 1,
                    'name' => $args[0],
                ];

                $posts = get_posts( $query );
                if ( empty( $posts ) )
                    \WP_CLI::error( 'Post not found' );
                else
                    print_r( $this->plugin->get_plugin_media( $posts[0] ) );
            }
        }

        \WP_CLI::add_command( 'rswp', new PluginSiteCLI() );
    }
} );
